#include <ADS131A04 ADC/ADS131A04.h>

const eUSCI_SPI_MasterConfig adcSpiConfig =
{
        EUSCI_B_SPI_CLOCKSOURCE_SMCLK,             // SMCLK clock source
        24000000,                                  // SMCLK = 24 MHz
        24000000,                                  // SPICLK = 24 MHz (ADC maximum 25 MHz)
        EUSCI_B_SPI_MSB_FIRST,                     // MSB First
        EUSCI_B_SPI_PHASE_DATA_CHANGED_ONFIRST_CAPTURED_ON_NEXT, // Phase polarity (CPHA = 1); SPI mode 1
		EUSCI_B_SPI_CLOCKPOLARITY_INACTIVITY_LOW,  // Low polarity (CPOL = 0); SPI mode 1
		EUSCI_B_SPI_3PIN                           // 3 wire SPI mode
};

unsigned char rxData[ADS131A04_MAX_RX_BYTES];
char ADS131A04_status[ADS131A04_STATUS_STRING_LENGTH];
volatile bool adcTxFlag = false;
volatile bool adcRxFlag = false;
volatile bool adcSampleAvailable = false;
volatile bool adcChannelsEnabled = false;

bool ADS131A04_init(void)
{
	// Configuring GPIO
    MAP_GPIO_setAsInputPin(GPIO_PORT_P4, GPIO_PIN1); // !DONE
    MAP_GPIO_setAsInputPin(GPIO_PORT_P4, GPIO_PIN0); // !DRDY; input in case of asynchronous interrupt mode
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P1, GPIO_PIN4); // !ADC_SPI_CS
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P5, GPIO_PIN0); // M0
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P5, GPIO_PIN1); // M1
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P5, GPIO_PIN2); // M2
    // Pini care controleaza circuite analogice de pe placa, dar inclusi aici pentru ca efectuarea citirilor analogice este legata de ei
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P5, GPIO_PIN3); // AO_EN
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P8, GPIO_PIN7); // REG_EN

    // Setting GPIO states
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P5, GPIO_PIN0); // M0 = 1; asynchronous interrupt mode
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P5, GPIO_PIN1); // M1 = 1; SPI word size 32 bits
    MAP_GPIO_setOutputLowOnPin(GPIO_PORT_P5, GPIO_PIN2); // M2 = 0; Hamming code word validation off
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P1, GPIO_PIN4); // !ADC_SPI_CS = 1
    MAP_GPIO_setOutputLowOnPin(GPIO_PORT_P5, GPIO_PIN3); // AO_EN = 0; disable board differential opamps
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P8, GPIO_PIN7); // REG_EN = 1; enable +2V5 voltage regulators

    Timer32_sleep_cycles(4800000); // 100 ms @ MCLK = 48 MHz

    // Configuring SPI pins
    MAP_GPIO_setAsPeripheralModuleFunctionOutputPin(GPIO_PORT_P1, GPIO_PIN5, GPIO_PRIMARY_MODULE_FUNCTION); // ADC_SPI_SCLK
    MAP_GPIO_setAsPeripheralModuleFunctionOutputPin(GPIO_PORT_P1, GPIO_PIN6, GPIO_PRIMARY_MODULE_FUNCTION); // ADC_SPI_MOSI
    MAP_GPIO_setAsPeripheralModuleFunctionOutputPin(GPIO_PORT_P1, GPIO_PIN7, GPIO_PRIMARY_MODULE_FUNCTION); // ADC_SPI_MISO

    // Configuring SPI peripheral
    MAP_SPI_initMaster(EUSCI_B0_BASE, &adcSpiConfig);
    MAP_SPI_enableModule(EUSCI_B0_BASE);
    //MAP_SPI_enableInterrupt(EUSCI_B0_BASE, EUSCI_B_SPI_TRANSMIT_INTERRUPT); // Nu se foloseste cand se lucreaza la frecvente mari
    //MAP_SPI_enableInterrupt(EUSCI_B0_BASE, EUSCI_B_SPI_RECEIVE_INTERRUPT); // Nu se foloseste cand se lucreaza la frecvente mari
    //MAP_Interrupt_enableInterrupt(INT_EUSCIB0);

    adcChannelsEnabled = false;

    // Wait for ADC to be ready
    if (!ADS131A04_waitForReadyWord())
        return false;

    if (!ADS131A04_unlock()) // To enable SPI interface
        return false;

    // Secventa pentru a reseta ADC-ul; de exemplu se reseteaza controlerul (din watchdog) si trebuie reinitializat ADC-ul
    if (!ADS131A04_wakeup()) // Daca a fost in stand by trebuie intai trezit
        return false;

    if (!ADS131A04_reset())
        return false;

    if (!ADS131A04_unlock()) // Necesar dupa reset
        return false;

	if (!ADS131A04_writeSettings())
	    return false;

	if (!ADS131A04_standby())
        return false;

	return true;
}

bool ADS131A04_waitForReadyWord(void)
{
    unsigned char *rxData = (unsigned char*)calloc(4, sizeof(unsigned char));
    int noTries = 0;
    bool ADC_ready = false;

    MAP_GPIO_setOutputLowOnPin(GPIO_PORT_P8, GPIO_PIN7); // REG_EN = 0; disable +2V5 voltage regulators
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P8, GPIO_PIN7); // REG_EN = 1; enable +2V5 voltage regulators

    do
    {
        Timer32_sleep_cycles(240000); // 5 ms @ MCLK = 48 MHz
        rxData = ADS131A04_read(4);
        if ((*rxData == 0xFF && *(rxData + 1) == ADS131A04_CHANNEL_COUNT))// ||
            //(*rxData == 0xFF && *(rxData + 1) == GetByte(((ADS131A04_CHANNEL_COUNT | 0xFF00) >> 1), 0)))
        {
            ADC_ready = true;
        }
        noTries++;
        if (noTries == 10)
        {
            free(rxData);
            return false;
        }
    }
    while (!ADC_ready);

    free(rxData);
    return true;
}

bool ADS131A04_writeSettings(void)
{
    // ADC settings
    if (!ADS131A04_wreg(ADS131A04_A_SYS_CFG_ADDRESS, ADS131A04_A_SYS_CFG_VALUE))
        return false;

    if (!ADS131A04_wreg(ADS131A04_D_SYS_CFG_ADDRESS, ADS131A04_D_SYS_CFG_VALUE))
        return false;

    if (!ADS131A04_wreg(ADS131A04_CLK1_ADDRESS, ADS131A04_CLK1_VALUE))
        return false;

    if (!ADS131A04_wreg(ADS131A04_CLK2_ADDRESS, ADS131A04_CLK2_VALUE))
        return false;

    if (!ADS131A04_wreg(ADS131A04_ADC1_ADDRESS, ADS131A04_ADC1_VALUE))
        return false;

    if (!ADS131A04_wreg(ADS131A04_ADC2_ADDRESS, ADS131A04_ADC2_VALUE))
        return false;

    if (!ADS131A04_wreg(ADS131A04_ADC3_ADDRESS, ADS131A04_ADC3_VALUE))
        return false;

    if (!ADS131A04_wreg(ADS131A04_ADC4_ADDRESS, ADS131A04_ADC4_VALUE))
        return false;

    return true;
}

bool ADS131A04_sample(unsigned int* readDataCh1, unsigned int* readDataCh2, unsigned int* readDataCh3, unsigned int* readDataCh4, int noSamples)
{
    int i;

    ADS131A04_enableAnalogCircuits();

    if (!ADS131A04_wakeup())
    {
        ADS131A04_disableAnalogCircuits();
        return false;
    }

    ADS131A04_enableSamplingInterrupt();

    unsigned char *rxData = (unsigned char*)calloc(20, sizeof(unsigned char));
    for (i = 0; i < ADS131A04_DISCARDED_SAMPLES; i++)
    {
        while (adcSampleAvailable == false) // Wait for !DRDY
        {
            //MAP_PCM_gotoLPM0InterruptSafe();
            //MAP_Interrupt_enableMaster();
        }
        adcSampleAvailable = false;
        rxData = ADS131A04_read(20);
    }
    for (i = 0; i < noSamples; i++)
    {
        while (adcSampleAvailable == false) // Wait for !DRDY
        {
            //MAP_PCM_gotoLPM0InterruptSafe();
            //MAP_Interrupt_enableMaster();
        }
        adcSampleAvailable = false;

        // Read samples
        rxData = ADS131A04_read(20);

        /*if (i % 20 != 0)
        {
            *(readDataCh1 + i) = 0xFFFFFF;
            *(readDataCh2 + i) = 0xFFFFFF;
            *(readDataCh3 + i) = 0xFFFFFF;
        }
        else
        {*/
            *(readDataCh1 + i) = (*(rxData + 4) << 16) | (*(rxData + 5) << 8) | (*(rxData + 6));
            *(readDataCh2 + i) = (*(rxData + 8) << 16) | (*(rxData + 9) << 8) | (*(rxData + 10));
            *(readDataCh3 + i) = (*(rxData + 12) << 16) | (*(rxData + 13) << 8) | (*(rxData + 14));
        //}
       *(readDataCh4 + i) = (*(rxData + 16) << 16) | (*(rxData + 17) << 8) | (*(rxData + 18));

        if (i == 500)
        {
            int a = 999;
        }
    }

    while (adcSampleAvailable == false) // Wait for !DRDY
    {
        //MAP_PCM_gotoLPM0InterruptSafe();
        //MAP_Interrupt_enableMaster();
    }
    adcSampleAvailable = false;

    if (!ADS131A04_standby())
    {
        free(rxData);
        ADS131A04_disableSamplingInterrupt();
        ADS131A04_disableAnalogCircuits();
        return false;
    }

    free(rxData);
    ADS131A04_disableSamplingInterrupt();
    ADS131A04_disableAnalogCircuits();

    return true;
}

bool ADS131A04_initSampling(void)
{
    ADS131A04_enableAnalogCircuits();

    if (!ADS131A04_wakeup())
    {
        ADS131A04_disableAnalogCircuits();
        return false;
    }

    ADS131A04_enableSamplingInterrupt();

    return true;
}

bool ADS131A04_stopSampling(void)
{
    if (!ADS131A04_standby())
    {
        ADS131A04_disableSamplingInterrupt();
        ADS131A04_disableAnalogCircuits();
        return false;
    }

    ADS131A04_disableSamplingInterrupt();
    ADS131A04_disableAnalogCircuits();

    return true;
}

bool ADS131A04_singleSample(unsigned int* readDataCh1, unsigned int* readDataCh2, unsigned int* readDataCh3, unsigned int* readDataCh4, bool sampleGeophones)
{
    unsigned char *rxData = (unsigned char*)calloc(20, sizeof(unsigned char));

    while (adcSampleAvailable == false) // Wait for !DRDY
    {
        //MAP_PCM_gotoLPM0InterruptSafe();
        //MAP_Interrupt_enableMaster();
    }
    adcSampleAvailable = false;

    // Read samples
    rxData = ADS131A04_read(20);
    if (sampleGeophones == false)
    {
        readDataCh1[0] = 0xFFFFFF;
        readDataCh2[0] = 0xFFFFFF;
        readDataCh3[0] = 0xFFFFFF;
    }
    else
    {
        readDataCh1[0] = (*(rxData + 4) << 16) | (*(rxData + 5) << 8) | (*(rxData + 6));
        readDataCh2[0] = (*(rxData + 8) << 16) | (*(rxData + 9) << 8) | (*(rxData + 10));
        readDataCh3[0] = (*(rxData + 12) << 16) | (*(rxData + 13) << 8) | (*(rxData + 14));
    }
    readDataCh4[0] = (*(rxData + 16) << 16) | (*(rxData + 17) << 8) | (*(rxData + 18));

    free(rxData);
    return true;
}

/**
 * @brief Citeste un singur esantion de la ADC.
 * Pregatirea si oprirea esantionarii se fac separat.
 * Datele se salveaza in buffere aferente fiecarui octet, pentru
 * a evita apelarea unei functii GetByte() pentru extragerea octetilor,
 * ceea ce ar creste timpul de lucru.
 * Functia se foloseste pentru a citi direct valorile esantioanelor in functiile urmatoare
 *
 * @param readDataCh1Byte1 Pointer catre bufferul aferent octetului 1 al canalului 1
 * @param readDataCh1Byte2 Pointer catre bufferul aferent octetului 2 al canalului 1
 * @param readDataCh1Byte3 Pointer catre bufferul aferent octetului 3 al canalului 1
 * @param readDataCh2Byte1 Pointer catre bufferul aferent octetului 1 al canalului 2
 * @param readDataCh2Byte2 Pointer catre bufferul aferent octetului 2 al canalului 2
 * @param readDataCh2Byte3 Pointer catre bufferul aferent octetului 3 al canalului 2
 * @param readDataCh3Byte1 Pointer catre bufferul aferent octetului 1 al canalului 3
 * @param readDataCh3Byte2 Pointer catre bufferul aferent octetului 2 al canalului 3
 * @param readDataCh3Byte3 Pointer catre bufferul aferent octetului 3 al canalului 3
 * @param readDataCh4Byte1 Pointer catre bufferul aferent octetului 1 al canalului 4
 * @param readDataCh4Byte2 Pointer catre bufferul aferent octetului 2 al canalului 4
 * @param readDataCh4Byte3 Pointer catre bufferul aferent octetului 3 al canalului 4
 * @param sampleGeophones Flag prin care se specifica daca se doreste esantioanarea geofoanelor; in cazul false, completeaza cu 0xFFFFFF esantioanele geofoanelor
 *
 * @return true/false daca esantionarea a avut succes/a esuat
**/
bool ADS131A04_singleSampleFast(unsigned char* readDataCh1Byte1, unsigned char* readDataCh1Byte2, unsigned char* readDataCh1Byte3,
                                unsigned char* readDataCh2Byte1, unsigned char* readDataCh2Byte2, unsigned char* readDataCh2Byte3,
                                unsigned char* readDataCh3Byte1, unsigned char* readDataCh3Byte2, unsigned char* readDataCh3Byte3,
                                unsigned char* readDataCh4Byte1, unsigned char* readDataCh4Byte2, unsigned char* readDataCh4Byte3,
                                bool sampleGeophones)
{
    unsigned char *rxData = (unsigned char*)calloc(20, sizeof(unsigned char));

    while (adcSampleAvailable == false) // Wait for !DRDY
    {
        //MAP_PCM_gotoLPM0InterruptSafe();
        //MAP_Interrupt_enableMaster();
    }
    adcSampleAvailable = false;

    // Read samples
    rxData = ADS131A04_read(20); // STAT_1 x 4 octeti + 4 canale x 4 octeti
    if (sampleGeophones == false) // Geofoanele se esantioneaza la 512 Hz; decimare cu 20
    { // Se neglijeaza ultimul octet ca esantioanele sunt pe 24 biti (3 octeti)
        //readDataCh1[j] = ((*(rxData + 4) << 24) | (*(rxData + 5) << 16) | (*(rxData + 6) << 8) | (*(rxData + 7))) >> 8; // Channel 1 - Geophone 1
        //readDataCh2[j] = ((*(rxData + 8) << 24) | (*(rxData + 9) << 16) | (*(rxData + 10) << 8) | (*(rxData + 11))) >> 8; // Channel 2 - Geophone 2
        //readDataCh3[j] = ((*(rxData + 12) << 24) | (*(rxData + 13) << 16) | (*(rxData + 14) << 8) | (*(rxData + 15))) >> 8; // Channel 3 - Geophone 3
        readDataCh1Byte1[0] = 0xFF; readDataCh1Byte2[0] = 0xFF; readDataCh1Byte3[0] = 0xFF; // Channel 1 - Geophone 1
        readDataCh2Byte1[0] = 0xFF; readDataCh2Byte2[0] = 0xFF; readDataCh2Byte3[0] = 0xFF; // Channel 2 - Geophone 2
        readDataCh3Byte1[0] = 0xFF; readDataCh3Byte2[0] = 0xFF; readDataCh3Byte3[0] = 0xFF; // Channel 3 - Geophone 3
    }
    else
    {
        //readDataCh1[0] = (*(rxData + 4) << 16) | (*(rxData + 5) << 8) | (*(rxData + 6)); // Channel 1 - Geophone 1
        readDataCh1Byte1[0] = *(rxData + 4); readDataCh1Byte2[0] = *(rxData + 5); readDataCh1Byte3[0] = *(rxData + 6); // Channel 1 - Geophone 1
        //readDataCh2[0] = (*(rxData + 8) << 16) | (*(rxData + 9) << 8) | (*(rxData + 10)); // Channel 2 - Geophone 2
        readDataCh2Byte1[0] = *(rxData + 8); readDataCh2Byte2[0] = *(rxData + 9); readDataCh2Byte3[0] = *(rxData + 10); // Channel 2 - Geophone 2
        //readDataCh3[0] = (*(rxData + 12) << 16) | (*(rxData + 13) << 8) | (*(rxData + 14)); // Channel 3 - Geophone 3
        readDataCh3Byte1[0] = *(rxData + 12); readDataCh3Byte2[0] = *(rxData + 13); readDataCh3Byte3[0] = *(rxData + 14); // Channel 3 - Geophone 3
    }
    // Microfonul se esantioneaza la 10.240 kHz
    //readDataCh4[i] = ((*(rxData + 16) << 24) | (*(rxData + 17) << 16) | (*(rxData + 18) << 8) | (*(rxData + 19))) >> 8; // Channel 4 - Microphone
    //readDataCh4[0] = (*(rxData + 16) << 16) | (*(rxData + 17) << 8) | (*(rxData + 18)); // Channel 4 - Microphone
    readDataCh4Byte1[0] = *(rxData + 16); readDataCh4Byte2[0] = *(rxData + 17); readDataCh4Byte3[0] = *(rxData + 18); // Channel 4 - Microphone

    free(rxData);
    return true;
}

bool ADS131A04_enableAnalogCircuits(void)
{
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P5, GPIO_PIN3); // AO_EN = 1
/*    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P8, GPIO_PIN7); // REG_EN = 1

    if (!ADS131A04_waitForReadyWord()) // Prima data raspunde 0xFF82, apoi 0xFF04; abia acum se pot da comenzi; de aceea am comentat ADS131A04_waitForReadyWord() a doua conditie din while, ca sa pot folosi functia si aici
        return false;

    if (!ADS131A04_unlock())
        return false;
*/
    return true;
}

void ADS131A04_disableAnalogCircuits(void)
{
    MAP_GPIO_setOutputLowOnPin(GPIO_PORT_P5, GPIO_PIN3); // AO_EN = 0
    //MAP_GPIO_setOutputLowOnPin(GPIO_PORT_P8, GPIO_PIN7); // REG_EN = 0; reseteaza ADC-ul si trebuie reinitializat la fiecare wakeup
}

void ADS131A04_enableSamplingInterrupt(void)
{
    MAP_GPIO_clearInterruptFlag(GPIO_PORT_P4, GPIO_PIN0);
    MAP_GPIO_interruptEdgeSelect(GPIO_PORT_P4, GPIO_PIN0, GPIO_HIGH_TO_LOW_TRANSITION);
    MAP_GPIO_enableInterrupt(GPIO_PORT_P4, GPIO_PIN0);
    MAP_Interrupt_enableInterrupt(INT_PORT4);
}

void ADS131A04_disableSamplingInterrupt(void)
{
    MAP_GPIO_clearInterruptFlag(GPIO_PORT_P4, GPIO_PIN0);
    MAP_GPIO_disableInterrupt(GPIO_PORT_P4, GPIO_PIN0);
}

/**
 * @brief Null Command. The NULL command has no effect on ADC registers or data.
 * Rather than producing an ACK response on DOUT, the command issues a register
 * readback of the STAT_1 register to monitor for general fault updates.
 *
 * @param readData Buffer pentru stocarea valorii registrului STAT_1
**/
void ADS131A04_null(unsigned char* readData)
{
    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "STAT_1: Status 1 register status:\n");

    if (adcChannelsEnabled == false)
    {
        ADS131A04_writeInt((ADS131A04_NULL_CMD << 16), 4);
    }
    else
    {
        ADS131A04_writeInt((ADS131A04_NULL_CMD << 16), 20);
    }

	unsigned char *rxData;
	if (adcChannelsEnabled == false)
    {
        rxData = ADS131A04_read(4);
    }
    else
    {
        rxData = ADS131A04_read(20);
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	if (GetBit(*rxData, 0) == 1) // F_CHECK; R; Reset 0h; Fault DIN check
	{
		// DIN transmission error
		/*
		 * This bit is set if either of the following conditions are detected:
		 * Uncorrectable hamming error correction state is determined for any DIN word transfer when hamming code is enabled.
		 * CRC check word on DIN fails. The input command that triggered this error is ignored.
		 *
		 * This bit auto-clears on a STAT_S transfer, unless the condition remains.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_CHECK; R; Reset 0h; Fault DIN check: DIN transmission error\n");
	}
	else
	{
		// No error in DIN transmission
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_CHECK; R; Reset 0h; Fault DIN check: No error in DIN transmission\n");
	}
	if (GetBit(*rxData, 1) == 1) // F_DRDY; R; Reset 0h; Fault data ready
	{
		// New data update during DOUT data transmission
		/*
		 * This bit is set if data shifted out from the previous result are not complete
		 * by the time new ADC data are ready. The ADC DRDY line pulses,
		 * indicating that new data are available and overwrite the current data. This
		 * bit auto-clears on a STAT_1 transfer, unless the condition remains.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_DRDY; R; Reset 0h; Fault data ready: New data update during DOUT data transmission\n");
	}
	else
	{
		// Data read back complete before new data update
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_DRDY; R; Reset 0h; Fault data ready: Data read back complete before new data update\n");
	}
	if (GetBit(*rxData, 2) == 1) // F_RESYNC; R; Reset 0h; Fault resynchronization
	{
		// Signal path is momentarily reset to maintain synchronization
		/*
		 * This bit is set whenever the signal path is momentarily reset resulting from
		 * a DRDY synchronization event.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_RESYNC; R; Reset 0h; Fault resynchronization: Signal path is momentarily reset to maintain synchronization\n");
	}
	else
	{
		// Devices are in sync
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_RESYNC; R; Reset 0h; Fault resynchronization: Devices are in sync\n");
	}
	if (GetBit(*rxData, 3) == 1) // F_WDT; R; Reset 0h; Watchdog timer timeout
	{
		// Timer has run out (resets following register read back)
		/*
		 * This bit indicates if the watchdog timer times out before a new data frame
		 * transfer occurs.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_WDT; R; Reset 0h; Watchdog timer timeout: Timer has run out (resets following register read back)\n");
	}
	else
	{
		// No fault has occurred regarding watchdog timer timeout
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_WDT; R; Reset 0h; Watchdog timer timeout: No fault has occurred regarding watchdog timer timeout\n");
	}
	if (GetBit(*rxData, 4) == 1) // F_ADCIN; R; Reset 0h; Fault ADC input
	{
		// A bit in the STAT_P or STAT_N register is set high
		/*
		 * This bit indicates that one of the ADC input fault detection bits in the
		 * STAT_P or STAT_N register is set. Read the STAT_P and STAT_N
		 * registers to clear the bit.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_ADCIN; R; Reset 0h; Fault ADC input: A bit in the STAT_P or STAT_N register is set high\n");
	}
	else
	{
		// No fault has occurred regarding ADC input
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_ADCIN; R; Reset 0h; Fault ADC input: No fault has occurred regarding ADC input\n");
	}
	if (GetBit(*rxData, 5) == 1) // F_SPI; R; Reset 0h; Fault SPI
	{
		// A bit in the STAT_S register is set high
		/*
		 * This bit indicates that one of the status bits in the STAT_S register is set.
		 * Read the STAT_S register to clear the bit.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_SPI; R; Reset 0h; Fault SPI: A bit in the STAT_S register is set high\n");
	}
	else
	{
		// No fault has occurred regarding SPI
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_SPI; R; Reset 0h; Fault SPI: No fault has occurred regarding SPI\n");
	}
	if (GetBit(*rxData, 6) == 1) // F_OPC; R; Reset 0h; Fault command
	{
		// Possible invalid command is ignored
		/*
		 * This bit indicates that a received command is not recognized as valid and
		 * the command is ignored. This bit auto-clears on a STAT_1 data transfer,
		 * unless the condition remains.
		 * When in a locked state, this bit is set if any command other than LOCK,
		 * UNLOCK, NULL, or RREGS is written to the device.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_OPC; R; Reset 0h; Fault command: Possible invalid command is ignored\n");
	}
	else
	{
		// No fault has occurred regarding commands
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_OPC; R; Reset 0h; Fault command: No fault has occurred regarding commands\n");
	}

	// Bit 7 reserved. Always read 0.

	//if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = *rxData;
}

/**
 * @brief Reset to POR Values. The RESET command places the ADC into a power-on reset
 * (POR) state, resetting all user registers to the default states. The reset
 * begins following the completion of the frame. When reset completes, the
 * ADC enters a reset locked state and outputs the READY status response on
 * DOUT as the command status response.
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_reset(void/*unsigned char* readData*/)
{
	ADS131A04_writeInt((ADS131A04_RESET_CMD << 16), 4);

	// Timp pentru repornire
	Timer32_sleep_cycles(240000); // 5 ms @ MCLK = 48 MHz

    unsigned char *rxData = (unsigned char*)calloc(4, sizeof(unsigned char));
	rxData = ADS131A04_read(4);
	if (*rxData == 0xFF && *(rxData + 1) == ADS131A04_CHANNEL_COUNT ||
	    *rxData == 0xFF && *(rxData + 1) == GetByte(((ADS131A04_CHANNEL_COUNT | 0xFF00) >> 1), 0))
	{
		// Software reset successful
	    //*readData = *(rxData + 1); // Device ID
	    adcChannelsEnabled = false;
	    free(rxData);
	    return true;
	}
	else
	{
		// Software reset unsuccessful
	    free(rxData);
	    return false;
	}
}

/**
 * @brief Enter Standby Mode. The STANDBY command places the ADC in a low-power
 * standby mode, halting conversions. The digital interface remains powered,
 * allowing all registers to retain the previous states. When in standby mode,
 * writing and reading from registers is possible and any programmable bits
 * that activate circuitry take effect in the device after the WAKEUP command
 * is issued. The command status response following a STANDBY command is 0022h.
 * In standby mode, the command status response is dependent on the user
 * command that is sent. All ADC channels must be disabled (by writing to the
 * ADCx registers) prior to entering standby mode to reduce current consumption.
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_standby(void)
{
	// Dezactivare canale ADC inaintea intrarii in stand by pentru reducerea curentului consumat
    if (!ADS131A04_write_ENA(ADS131A04_ENA_POWER_DOWN))
        return false;

    ADS131A04_writeInt((ADS131A04_STANDBY_CMD << 16), 4);
    unsigned char *rxData = (unsigned char*)calloc(4, sizeof(unsigned char));
	rxData = ADS131A04_read(4);
	if (*(rxData + 1) == 0x22)
	{
		// Enter standby mode successful
	    free(rxData);
	    return true;
	}
	else
	{
		// Enter standby mode unsuccessful
	    free(rxData);
	    return false;
	}
}

/**
 * @brief Exit Standby Mode. The WAKEUP command brings the ADC out of standby
 * mode. The ADC channels must be enabled (by writing to the ADCx registers)
 * before bringing the device out of standby mode. Allow enough time for all
 * circuits in standby mode to power-up (see the Electrical Characteristics
 * table for details). The command status response following a WAKEUP command
 * is 0033h.
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_wakeup(void)
{
    // Activare canale ADC inaintea iesirii din stand by
    if (!ADS131A04_write_ENA(ADS131A04_ENA_POWER_UP))
        return false;

    if (adcChannelsEnabled == false)
    {
        ADS131A04_writeInt((ADS131A04_WAKEUP_CMD << 16), 4);
    }
    else
    {
        ADS131A04_writeInt((ADS131A04_WAKEUP_CMD << 16), 20);
    }

    unsigned char *rxData = (unsigned char*)calloc(20, sizeof(unsigned char));
    if (adcChannelsEnabled == false)
    {
        rxData = ADS131A04_read(4);
    }
    else
    {
        rxData = ADS131A04_read(20);
    }
	if (*(rxData + 1) == 0x33)
	{
		// Exit standby mode successful
	    // Astepta timp pornire Vref cand se comuta la referinta interna
	    //Timer32_sleep_cycles(12000000); // 250 ms @ MCLK = 48 MHz
	    free(rxData);
	    return true;
	}
	else
	{
		// Exit standby mode unsuccessful
	    free(rxData);
	    return false;
	}
}

/**
 * @brief Lock ADC Registers. The LOCK command places the converter interface
 * in a locked state where the interface becomes unresponsive to most input
 * commands. The UNLOCK, NULL, RREG, and RREGS commands are the only commands
 * that are recognized when reading back data. Following the LOCK command, the
 * first DOUT status response reads 0555h followed by the command status
 * response of a NULL command (by reading the STAT_1 register).
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_lock(void/*unsigned char* readData*/)
{
    if (adcChannelsEnabled == false)
    {
        ADS131A04_writeInt((ADS131A04_LOCK_CMD << 16), 4);
    }
    else
    {
        ADS131A04_writeInt((ADS131A04_LOCK_CMD << 16), 20);
    }

    unsigned char *rxData = (unsigned char*)calloc(20, sizeof(unsigned char));
	if (adcChannelsEnabled == false)
    {
        rxData = ADS131A04_read(4);
    }
    else
    {
        rxData = ADS131A04_read(20);
    }
	if ((*rxData == 0x05) && (*(rxData + 1) == 0x55))
	{
		// Lock command successful
		// *(rxData + 2) represents STAT_1 register (see null command)
	    //*readData = *(rxData + 2);
	    free(rxData);
	    return true;
	}
	else
	{
		// Lock command unsuccessful
	    free(rxData);
	    return false;
	}
}

/**
 * @brief Unlock ADC Registers. The UNLOCK command brings the converter out
 * of the locked state, allowing all registers to be accessed in the next
 * data frame. The command status response associated with the UNLOCK command
 * is 0655h.
 * When powering up the device or coming out of a power-on reset (POR) state,
 * the ADC is in a power-up ready state. In this mode the command status
 * response reads back FFDDh (DD denotes the channel count defined by the
 * NU_CH[3:0] bits in the ID_MSB register), indicating that the ADC power-on
 * cycle is complete and that the ADC is ready to accept commands. Use the
 * UNLOCK command to enable the SPI interface and begin communication with
 * the device. The command status response associated with the UNLOCK command
 * is 0655h.
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_unlock(void/*unsigned char* readData*/)
{
    if (adcChannelsEnabled == false)
    {
        ADS131A04_writeInt((ADS131A04_UNLOCK_CMD << 16), 4);
    }
    else
    {
        ADS131A04_writeInt((ADS131A04_UNLOCK_CMD << 16), 20);
    }

    unsigned char *rxData = (unsigned char*)calloc(20, sizeof(unsigned char));
	if (adcChannelsEnabled == false)
    {
        rxData = ADS131A04_read(4);
    }
    else
    {
        rxData = ADS131A04_read(20);
    }

	if ((*rxData == 0x06) && (*(rxData + 1) == 0x55))
	{
		// Unlock command successful
	    free(rxData);
	    return true;
	}
	else
	{
		/*if ((*rxData == 0xFF) && (*(rxData + 1) == ADS131A04_CHANNEL_COUNT))
		{
			// Unlock command successful; powering up the device or coming out of a power-on reset (POR) state
		    *readData = *(rxData + 1); //The channel count defined by the NU_CH[3:0] bits in the ID_MSB register
			return true;
		}
		else
		{
			// Unlock command unsuccessful
		    return false;
		}*/
	    free(rxData);
	    return false;
	}
}

/**
 * @brief Citeste numarul de revizie al ADC-ului.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului ID_LSB
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_RevisionID(unsigned char* readData)
{
    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_ID_LSB_ADDRESS, &rxData) == false)
    {
        return false;
    }
	/*
	 * These bits indicate the revision of the device and are subject ot change
	 * without notice.
	 * */

    /*if (ADS131A04_ADS131A04_getStatus == true) // Comentat pentru optimizare (sprintf consuma multe resurse)
    {
        strcat(ADS131A04_status, "ID_LSB_ADDRESS: Device revision ID register status:\nREV_ID[7:0]; R; Revision ID: ");
        unsigned char number[4];
        sprintf(number,"%d\n\n", rxData);
        strcat(ADS131A04_status, number);
    }*/

    *readData = rxData; // REV_ID[7:0]; R; Revision ID

    return true;
}

/**
 * @brief Citeste registrul STAT_1: Status 1 Register (address = 02h).
 * This register contains general fault updates. This register
 * is automatically transferred on the command status
 * response when the NULL command is sent.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului STAT_1
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_STAT_1(unsigned char* readData)
{
    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "STAT_1: Status 1 register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_STAT_1_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
    if (GetBit(rxData, 0) == 1) // F_CHECK; R; Reset 0h; Fault DIN check
    {
        // DIN transmission error
        /*
         * This bit is set if either of the following conditions are detected:
         * Uncorrectable hamming error correction state is determined for any DIN word transfer when hamming code is enabled.
         * CRC check word on DIN fails. The input command that triggered this error is ignored.
         *
         * This bit auto-clears on a STAT_S transfer, unless the condition remains.
         * */
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_CHECK; R; Reset 0h; Fault DIN check: DIN transmission error\n");
    }
    else
    {
        // No error in DIN transmission
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_CHECK; R; Reset 0h; Fault DIN check: No error in DIN transmission\n");
    }
    if (GetBit(rxData, 1) == 1) // F_DRDY; R; Reset 0h; Fault data ready
    {
        // New data update during DOUT data transmission
        /*
         * This bit is set if data shifted out from the previous result are not complete
         * by the time new ADC data are ready. The ADC DRDY line pulses,
         * indicating that new data are available and overwrite the current data. This
         * bit auto-clears on a STAT_1 transfer, unless the condition remains.
         * */
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_DRDY; R; Reset 0h; Fault data ready: New data update during DOUT data transmission\n");
    }
    else
    {
        // Data read back complete before new data update
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_DRDY; R; Reset 0h; Fault data ready: Data read back complete before new data update\n");
    }
    if (GetBit(rxData, 2) == 1) // F_RESYNC; R; Reset 0h; Fault resynchronization
    {
        // Signal path is momentarily reset to maintain synchronization
        /*
         * This bit is set whenever the signal path is momentarily reset resulting from
         * a DRDY synchronization event.
         * */
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_RESYNC; R; Reset 0h; Fault resynchronization: Signal path is momentarily reset to maintain synchronization\n");
    }
    else
    {
        // Devices are in sync
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_RESYNC; R; Reset 0h; Fault resynchronization: Devices are in sync\n");
    }
    if (GetBit(rxData, 3) == 1) // F_WDT; R; Reset 0h; Watchdog timer timeout
    {
        // Timer has run out (resets following register read back)
        /*
         * This bit indicates if the watchdog timer times out before a new data frame
         * transfer occurs.
         * */
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_WDT; R; Reset 0h; Watchdog timer timeout: Timer has run out (resets following register read back)\n");
    }
    else
    {
        // No fault has occurred regarding watchdog timer timeout
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_WDT; R; Reset 0h; Watchdog timer timeout: No fault has occurred regarding watchdog timer timeout\n");
    }
    if (GetBit(rxData, 4) == 1) // F_ADCIN; R; Reset 0h; Fault ADC input
    {
        // A bit in the STAT_P or STAT_N register is set high
        /*
         * This bit indicates that one of the ADC input fault detection bits in the
         * STAT_P or STAT_N register is set. Read the STAT_P and STAT_N
         * registers to clear the bit.
         * */
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_ADCIN; R; Reset 0h; Fault ADC input: A bit in the STAT_P or STAT_N register is set high\n");
    }
    else
    {
        // No fault has occurred regarding ADC input
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_ADCIN; R; Reset 0h; Fault ADC input: No fault has occurred regarding ADC input\n");
    }
    if (GetBit(rxData, 5) == 1) // F_SPI; R; Reset 0h; Fault SPI
    {
        // A bit in the STAT_S register is set high
        /*
         * This bit indicates that one of the status bits in the STAT_S register is set.
         * Read the STAT_S register to clear the bit.
         * */
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_SPI; R; Reset 0h; Fault SPI: A bit in the STAT_S register is set high\n");
    }
    else
    {
        // No fault has occurred regarding SPI
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_SPI; R; Reset 0h; Fault SPI: No fault has occurred regarding SPI\n");
    }
    if (GetBit(rxData, 6) == 1) // F_OPC; R; Reset 0h; Fault command
    {
        // Possible invalid command is ignored
        /*
         * This bit indicates that a received command is not recognized as valid and
         * the command is ignored. This bit auto-clears on a STAT_1 data transfer,
         * unless the condition remains.
         * When in a locked state, this bit is set if any command other than LOCK,
         * UNLOCK, NULL, or RREGS is written to the device.
         * */
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_OPC; R; Reset 0h; Fault command: Possible invalid command is ignored\n");
    }
    else
    {
        // No fault has occurred regarding commands
        //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_OPC; R; Reset 0h; Fault command: No fault has occurred regarding commands\n");
    }

    // Bit 7 reserved. Always read 0.

    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

    *readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul STAT_P: Positive Input Fault Detect Status Register
 * (address = 03h). This register stores the status of whether the positive input
 * on each channel exceeds the threshold set by the COMP_TH[2:0] bits.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului STAT_P
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_STAT_P(unsigned char* readData)
{
    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "STAT_P: Positive input fault detect status register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_STAT_P_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	if (GetBit(rxData, 0) == 1) // F_IN1P; R; Reset 0h; AIN1P threshold detect
	{
		// The channel 1 positive input pin exceeds the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN1P; R; Reset 0h; AIN1P threshold detect: The channel 1 positive input pin exceeds the set threshold\n");
	}
	else
	{
		// The channel 1 positive input pin does not exceed the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN1P; R; Reset 0h; AIN1P threshold detect: The channel 1 positive input pin does not exceed the set threshold\n");
	}
	if (GetBit(rxData, 1) == 1) // F_IN2P; R; Reset 0h; AIN2P threshold detect
	{
		// The channel 2 positive input pin exceeds the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN2P; R; Reset 0h; AIN2P threshold detect: The channel 2 positive input pin exceeds the set threshold\n");
	}
	else
	{
		// The channel 2 positive input pin does not exceed the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN2P; R; Reset 0h; AIN2P threshold detect: The channel 2 positive input pin does not exceed the set threshold\n");
	}
	if (GetBit(rxData, 2) == 1) // F_IN3P; R; Reset 0h; AIN3P threshold detect
	{
		// The channel 3 positive input pin exceeds the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN3P; R; Reset 0h; AIN3P threshold detect: The channel 3 positive input pin exceeds the set threshold\n");
	}
	else
	{
		// The channel 3 positive input pin does not exceed the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN3P; R; Reset 0h; AIN3P threshold detect: The channel 3 positive input pin does not exceed the set threshold\n");
	}
	if (GetBit(rxData, 3) == 1) // F_IN4P; R; Reset 0h; AIN4P threshold detect
	{
		// The channel 4 positive input pin exceeds the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN4P; R; Reset 0h; AIN4P threshold detect: The channel 4 positive input pin exceeds the set threshold\n");
	}
	else
	{
		// The channel 4 positive input pin does not exceed the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN4P; R; Reset 0h; AIN4P threshold detect: The channel 4 positive input pin does not exceed the set threshold\n");
	}

	// Bits 4-7 reserved. Always read 0.

	//if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul STAT_N: Negative Input Fault Detect Status
 * Register (address = 04h). This register stores the status of whether the
 * negative input on each channel exceeds the threshold set by the
 * COMP_TH[2:0] bits.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului STAT_N
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_STAT_N(unsigned char* readData)
{
    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "STAT_N: Negative input fault detect status register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_STAT_N_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	if (GetBit(rxData, 0) == 1) // F_IN1N; R; Reset 0h; AIN1N threshold detect
	{
		// The channel 1 negative input pin exceeds the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN1N; R; Reset 0h; AIN1N threshold detect: The channel 1 negative input pin exceeds the set threshold\n");
	}
	else
	{
		// The channel 1 negative input pin does not exceed the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN1N; R; Reset 0h; AIN1N threshold detect: The channel 1 negative input pin does not exceed the set threshold\n");
	}
	if (GetBit(rxData, 1) == 1) // F_IN2N; R; Reset 0h; AIN2N threshold detect
	{
		// The channel 2 negative input pin exceeds the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN2N; R; Reset 0h; AIN2N threshold detect: The channel 2 negative input pin exceeds the set threshold\n");
	}
	else
	{
		// The channel 2 negative input pin does not exceed the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN2N; R; Reset 0h; AIN2N threshold detect: The channel 2 negative input pin does not exceed the set threshold\n");
	}
	if (GetBit(rxData, 2) == 1) // F_IN3N; R; Reset 0h; AIN3N threshold detect
	{
		// The channel 3 negative input pin exceeds the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN3N; R; Reset 0h; AIN3N threshold detect: The channel 3 negative input pin exceeds the set threshold\n");
	}
	else
	{
		// The channel 3 negative input pin does not exceed the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN3N; R; Reset 0h; AIN3N threshold detect: The channel 3 negative input pin does not exceed the set threshold\n");
	}
	if (GetBit(rxData, 3) == 1) // F_IN4N; R; Reset 0h; AIN4N threshold detect
	{
		// The channel 4 negative input pin exceeds the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN4N; R; Reset 0h; AIN4N threshold detect: The channel 4 negative input pin exceeds the set threshold\n");
	}
	else
	{
		// The channel 4 negative input pin does not exceed the set threshold
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_IN4N; R; Reset 0h; AIN4N threshold detect: The channel 4 negative input pin does not exceed the set threshold\n");
	}

	// Bits 4-7 reserved. Always read 0.

	//if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul STAT_S: SPI Status Register (address = 05h).
 * This register indicates the detection of SPI fault conditions.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului STAT_S
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_STAT_S(unsigned char* readData)
{
    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "STAT_S: SPI status register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_STAT_S_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	if (GetBit(rxData, 0) == 1) // F_FRAME; R; Reset 0h; Frame fault
	{
		// Not enough SCLKs are sent per frame
		/*
		 * This bit is set if the device detects that not enough SCLK cycles are sent in
		 * a data frame for the existing mode of operation. This bit auto-clears on a
		 * STAT_S transfer, unless the condition remains.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_FRAME; R; Reset 0h; Frame fault: Not enough SCLKs are sent per frame\n");
	}
	else
	{
		// Enough SCLKs are sent per frame
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_FRAME; R; Reset 0h; Frame fault: Enough SCLKs are sent per frame\n");
	}
	if (GetBit(rxData, 1) == 1) // F_CS; R; Reset 0h; Chip-select fault
	{
		// CS is asserted or deasserted when SCLK is high
		/*
		 * This bit is set if CS transitions when the SCLK pin is high. This bit autoclears
		 * on a STAT_S transfer, unless the condition remains.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_CS; R; Reset 0h; Chip-select fault: CS is asserted or deasserted when SCLK is high\n");
	}
	else
	{
		// CS is asserted or deasserted when SCLK is low
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_CS; R; Reset 0h; Chip-select fault: CS is asserted or deasserted when SCLK is low\n");
	}
	if (GetBit(rxData, 2) == 1) // F_STARTUP; R; Reset 0h; ADC startup fault
	{
		// A fault has occurred
		/*
		 * This bit indicates if an error is detected during power-up. This bit clears
		 * only when power is recycled.
		 * */
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_STARTUP; R; Reset 0h; ADC startup fault: A fault has occurred\n");
	}
	else
	{
		// No fault is occurred
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "F_STARTUP; R; Reset 0h; ADC startup fault: No fault is occurred\n");
	}

	// Bits 3-7 reserved. Always read 0.

	//if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul ERROR_CNT: Error Count Register (address = 06h)
 * This register counts the hamming and CRC errors. This register is cleared
 * when read.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului ERROR_CNT
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_ErrorCount(unsigned char* readData)
{
    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_ERROR_CNT_ADDRESS, &rxData) == false)
    {
        return false;
    }

    /*if (ADS131A04_ADS131A04_getStatus == true) // Comentat pentru optimizare (sprintf consuma multe resurse)
    {
        strcat(ADS131A04_status, "ERROR_CNT: Error count register status:\nER[7:0]; R/W; Reset 0h; Error tracking count: ");
        unsigned char number[3];
        sprintf(number,"%d\n\n", rxData);
        strcat(ADS131A04_status, number);
    }*/

    *readData = rxData; // ER[7:0]; R/W; Reset 0h; Error tracking count
	/*
	 * These bits count the number of hamming and CRC errors on the input. The
	 * counter saturates if the number of errors exceeds 255, FFh. This register is
	 * cleared when read.
	 * */

    return true;
}

/**
 * @brief Citeste registrul STAT_M2: Hardware Mode Pin Status Register
 * (address = 07h).
 * This register indicates detection of the captured states of the
 * hardware mode pins.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului STAT_M2
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_STAT_M2(unsigned char* readData)
{
    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "STAT_M2: Hardware mode pin status register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_STAT_M2_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	switch (GetBitsFromChar(rxData, 2, 0)) // M0PIN[1:0]; R; M0 captured state
	{
		/*
		 * These bits indicate the captured states of the
		 * M0 hardware control pins.
		 * */
        case 0:
			// GND (synchronous master mode)
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M0PIN[1:0]; R; M0 captured state: GND (synchronous master mode)\n");
			break;
		case 1:
			// IOVDD (asynchronous slave mode)
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M0PIN[1:0]; R; M0 captured state: IOVDD (asynchronous slave mode)\n");
			break;
		case 2:
			// No connection (synchronous slave mode)
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M0PIN[1:0]; R; M0 captured state: No connection (synchronous slave mode)\n");
			break;
		case 3:
			// Reserved
			break;
	}
	switch (GetBitsFromChar(rxData, 2, 2)) // M1PIN[1:0]; R; M1 captured state
	{
		/*
		 * These bits indicate the captured states of the
		 * M1 hardware control pins.
		 * */
        case 0:
			// GND (24-bit device word)
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M1PIN[1:0]; R; M1 captured state: GND (24-bit device word)\n");
			break;
		case 1:
			// IOVDD (32-bit device word)
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M1PIN[1:0]; R; M1 captured state: IOVDD (32-bit device word)\n");
			break;
		case 2:
			// No connection (16-bit device word)
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M1PIN[1:0]; R; M1 captured state: No connection (16-bit device word)\n");
			break;
		case 3:
			// Reserved
			break;
	}
	switch (GetBitsFromChar(rxData, 2, 4)) // M2PIN[1:0]; R; M2 captured state
	{
		/*
		 * These bits indicate the captured states of the
		 * M2 hardware control pins.
		 * */
        case 0:
			// GND (hamming code word validation off)
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M2PIN[1:0]; R; M2 captured state: GND (hamming code word validation off)\n");
			break;
		case 1:
			// IOVDD (hamming code word validation on)
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M2PIN[1:0]; R; M2 captured state: IOVDD (hamming code word validation on)\n");
			break;
		case 2:
			// No connection
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "M2PIN[1:0]; R; M2 captured state: No connection\n");
			break;
		case 3:
			// Reserved
			break;
	}

	// Bits 6-7 reserved. Always read 0.

	//if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul A_SYS_CFG: Analog System Configuration Register
 * (address = 0Bh).
 * This register configures the analog features in the ADS131A0x.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului A_SYS_CFG
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_A_SYS_CFG(unsigned char* readData)
{
    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "A_SYS_CFG: Analog system configuration register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_A_SYS_CFG_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	switch (GetBitsFromChar(rxData, 3, 0)) // COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold
	{
		/*
		 * These bits determine the fault detect comparator threshold level settings.
		 * When VNCPEN = 0, AVDD and AVSS are used for the high and low
		 * threshold. When VNCPEN = 1, AVDD is used for the high threshold value.
		 * A 1.5-V supply, generated from the negative charge pump, is used for the
		 * low threshold value.
		 * */
        case 0:
			// Comparator high-side threshold (%): 95
			// Comparator low-side threshold (%): 5
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold: Comparator high-side/low-side threshold (%): 95/5\n");
			break;
		case 1:
			// Comparator high-side threshold (%): 92.5
			// Comparator low-side threshold (%): 7.5
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold: Comparator high-side/low-side threshold (%): 92.5/7.5\n");
			break;
		case 2:
			// Comparator high-side threshold (%): 90
			// Comparator low-side threshold (%): 10
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold: Comparator high-side/low-side threshold (%): 90/10\n");
			break;
		case 3:
			// Comparator high-side threshold (%): 87.5
			// Comparator low-side threshold (%): 12.5
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold: Comparator high-side/low-side threshold (%): 87.5/12.5\n");
			break;
		case 4:
			// Comparator high-side threshold (%): 85
			// Comparator low-side threshold (%): 15
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold: Comparator high-side/low-side threshold (%): 85/15\n");
			break;
		case 5:
			// Comparator high-side threshold (%): 80
			// Comparator low-side threshold (%): 20
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold: Comparator high-side/low-side threshold (%): 80/20\n");
			break;
		case 6:
			// Comparator high-side threshold (%): 75
			// Comparator low-side threshold (%): 25
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold: Comparator high-side/low-side threshold (%): 75/25\n");
			break;
		case 7:
			// Comparator high-side threshold (%): 70
			// Comparator low-side threshold (%): 30
		    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "COMP_TH[2:0]; R/W; Reset 0h; Fault detect comparator threshold: Comparator high-side/low-side threshold (%): 70/30\n");
			break;
	}
	if (GetBit(rxData, 3) == 1) // INT_REFEN; R/W; Reset 0h; Enable internal reference
	{
		/*
		 * This bit connects the internal reference voltage to the
		 * reference buffer to use the internal reference
		 * */
		// Internal reference voltage enabled
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "INT_REFEN; R/W; Reset 0h; Enable internal reference: Internal reference voltage enabled\n");
	}
	else
	{
		// External reference voltage
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "INT_REFEN; R/W; Reset 0h; Enable internal reference: External reference voltage\n");
	}
	if (GetBit(rxData, 4) == 1) // VREF_4V; R/W; Reset 0h; REFP reference voltage level
	{
		/*
		 * This bit determines the REFP reference voltage level
		 * when using the internal reference.
		 * */
		// REFP is set to 4.0 V
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "VREF_4V; R/W; Reset 0h; REFP reference voltage level: REFP is set to 4.0 V\n");
	}
	else
	{
		// REFP is set to 2.442 V
	    //if (ADS131A04_ADS131A04_getStatus == true) strcat(ADS131A04_status, "VREF_4V; R/W; Reset 0h; REFP reference voltage level: REFP is set to 2.442 V\n");
	}

	// Bit 5 reserved. Always write 0.

	if (GetBit(rxData, 6) == 1) // HRM; R/W; Reset 1h; High-resolution mode
	{
		/*
		 * This bit selects between high-resolution and low-power mode.
		 * */
		// High-resolution mode
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "HRM; R/W; Reset 1h; High-resolution mode: High-resolution mode\n");
	}
	else
	{
		// Low-power mode
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "HRM; R/W; Reset 1h; High-resolution mode: Low-power mode\n");
	}
	if (GetBit(rxData, 7) == 1) // VNCPEN; R/W; Reset 0h; Negative charge pump enable
	{
		/*
		 * This bit enables the negative charge pump when using
		 * a 3.0-V to 3.45-V unipolar power supply.
		 * */
		// Negative charge pump enabled
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "VNCPEN; R/W; Reset 0h; Negative charge pump enable: Negative charge pump enabled\n");
	}
	else
	{
		// Negative charge pump powered down
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "VNCPEN; R/W; Reset 0h; Negative charge pump enable: Negative charge pump powered down\n");
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul D_SYS_CFG: Digital System Configuration Register
 * (address = 0Ch).
 * This register configures the digital features in the ADS131A0x.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului D_SYS_CFG
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_D_SYS_CFG(unsigned char* readData)
{
    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "D_SYS_CFG: Digital system configuration register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_D_SYS_CFG_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	if (GetBit(rxData, 0) == 1) // CRC_EN; R/W; Reset 0h; Cyclic redundancy check enable
	{
		/*
		 * This bit enables the CRC data word for both the DIN and DOUT
		 * data frame transfers. When enabled, DIN commands must pass the
		 * CRC checks to be recognized by the device.
		 * */
		// CRC enabled
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CRC_EN; R/W; Reset 0h; Cyclic redundancy check enable: CRC enabled\n");
	}
	else
	{
		// CRC disabled
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CRC_EN; R/W; Reset 0h; Cyclic redundancy check enable: CRC disabled\n");
	}
	if (GetBit(rxData, 1) == 1) // FIXED; R/W; Reset 0h; Fixed word size enable
	{
		/*
		 * This bit sets the data frame size.
		 * */
		// Fixed six device words per frame for the ADS131A04 or fixed four device words per data frame for the ADS131A02
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "FIXED; R/W; Reset 0h; Fixed word size enable: Fixed six device words per frame for the ADS131A04\n");
	}
	else
	{
		// Device words per data frame depends on whether the CRC and ADCs are enabled
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "FIXED; R/W; Reset 0h; Fixed word size enable: Device words per data frame depends on whether the CRC and ADCs are enabled\n");
	}
	switch (GetBitsFromChar(rxData, 2, 2)) // HIZDLY[1:0]; R/W; Reset 3h; Hi-Z delay
	{
		/*
		 * These bits configure the time that the device asserts
		 * Hi-Z on DOUT after the LSB of the data frame is shifted out.
		 * */
		case 0:
			// Greater or equal than 6-ns delay
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "HIZDLY[1:0]; R/W; Reset 3h; Hi-Z delay: Hi-Z delay: Greater or equal than 6-ns delay\n");
			break;
		case 1:
			// Greater or equal than 8-ns delay
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "HIZDLY[1:0]; R/W; Reset 3h; Hi-Z delay: Hi-Z delay: Greater or equal than 8-ns delay\n");
			break;
		case 2:
			// Greater or equal than 10-ns delay
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "HIZDLY[1:0]; R/W; Reset 3h; Hi-Z delay: Hi-Z delay: Greater or equal than 10-ns delay\n");
			break;
		case 3:
			// Greater or equal than 12-ns delay
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "HIZDLY[1:0]; R/W; Reset 3h; Hi-Z delay: Hi-Z delay: Greater or equal than 12-ns delay\n");
			break;
	}
	switch (GetBitsFromChar(rxData, 2, 4)) // DNDLY[1:0]; R/W; Reset 3h; DONE delay
	{
		/*
		 * These bits configure the time before the device asserts
		 * DONE after the LSB is shifted out.
		 * */
		case 0:
			// Greater or equal than 6-ns delay
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "DNDLY[1:0]; R/W; Reset 3h; DONE delay: DONE delay: Greater or equal than 6-ns delay\n");
			break;
		case 1:
			// Greater or equal than 8-ns delay
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "DNDLY[1:0]; R/W; Reset 3h; DONE delay: DONE delay: Greater or equal than 8-ns delay\n");
			break;
		case 2:
			// Greater or equal than 10-ns delay
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "DNDLY[1:0]; R/W; Reset 3h; DONE delay: DONE delay: Greater or equal than 10-ns delay\n");
			break;
		case 3:
			// Greater or equal than 12-ns delay
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "DNDLY[1:0]; R/W; Reset 3h; DONE delay: DONE delay: Greater or equal than 12-ns delay\n");
			break;
	}
	if (GetBit(rxData, 6) == 1) // CRC_MODE; R/W; Reset 0h; CRC mode select
	{
		/*
		 * This bit determines which bits in the frame the
		 * CRC is valid for.
		 * */
		// CRC is valid on all bits received and transmitted
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CRC_MODE; R/W; Reset 0h; CRC mode select: CRC is valid on all bits received and transmitted\n");
	}
	else
	{
		// CRC is valid on only the device words being sent and received
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CRC_MODE; R/W; Reset 0h; CRC mode select: CRC is valid on only the device words being sent and received\n");
	}
	if (GetBit(rxData, 7) == 1) // WDT_EN; R/W; Reset 0h; Watchdog timer enable
	{
		/*
		 * This bit enables the watchdog timeout counter when set.
		 * Issue a hardware or software reset when disabling the
		 * watchdog timer for internal device synchronization.
		 * */
		// Watchdog enabled
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "WDT_EN; R/W; Reset 0h; Watchdog timer enable: Watchdog enabled\n");
	}
	else
	{
		// Watchdog disabled
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "WDT_EN; R/W; Reset 0h; Watchdog timer enable: Watchdog disabled\n");
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul CLK1: Clock Configuration 1 Register
 * (address = 0Dh).
 * This register configures the ADC clocking and sets the internal clock dividers.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului CLK1
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_CLK1(unsigned char* readData)
{
    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK1: Clock configuration 1 register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_CLK1_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	// Bit 0 reserved. Always write 0.

	switch (GetBitsFromChar(rxData, 3, 1)) // CLK_DIV[2:0]; R/W; Reset 4h; CLKIN divider ratio
	{
		/*
		 * These bits set the CLKIN divider ratio to generate the
		 * internal fICLK frequency. ICLK is used as the fSCLK output
		 * when the ADC is operating in an SPI master mode.
		 * */
        case 0:
			// Reserved
			break;
		case 1:
			// fICLK = fCLKIN / 2
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK_DIV[2:0]; R/W; Reset 4h; CLKIN divider ratio: fICLK = fCLKIN / 2\n");
			break;
		case 2:
			// fICLK = fCLKIN / 4
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK_DIV[2:0]; R/W; Reset 4h; CLKIN divider ratio: fICLK = fCLKIN / 4\n");
			break;
		case 3:
			// fICLK = fCLKIN / 6
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK_DIV[2:0]; R/W; Reset 4h; CLKIN divider ratio: fICLK = fCLKIN / 6\n");
			break;
		case 4:
			// fICLK = fCLKIN / 8
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK_DIV[2:0]; R/W; Reset 4h; CLKIN divider ratio: fICLK = fCLKIN / 8\n");
			break;
		case 5:
			// fICLK = fCLKIN / 10
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK_DIV[2:0]; R/W; Reset 4h; CLKIN divider ratio: fICLK = fCLKIN / 10\n");
			break;
		case 6:
			// fICLK = fCLKIN / 12
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK_DIV[2:0]; R/W; Reset 4h; CLKIN divider ratio: fICLK = fCLKIN / 12\n");
			break;
		case 7:
			// fICLK = fCLKIN / 14
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK_DIV[2:0]; R/W; Reset 4h; CLKIN divider ratio: fICLK = fCLKIN / 14\n");
			break;
	}

	// Bits 4-6 reserved. Always write 0.

	if (GetBit(rxData, 7) == 1) // CLKSRC; R/W; Reset 0h; ADC clock source
	{
		/*
		 * This bit selects the source for ICLK; see the Clock
		 * section for more information on ADC clocking.
		 * */
		// SCLK pin
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLKSRC; R/W; Reset 0h; ADC clock source: SCLK pin\n");
	}
	else
	{
		// XTAL1/CLKIN pin or XTAL1/CLKIN and XTAL2 pins
	    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLKSRC; R/W; Reset 0h; ADC clock source: XTAL1/CLKIN pin or XTAL1/CLKIN and XTAL2 pins\n");
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul CLK2: Clock Configuration 2 Register
 * (address = 0Eh).
 * This register configure the ADC modulator clock and oversampling ratio for the converter.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului CLK2
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_CLK2(unsigned char* readData)
{
    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "CLK2: Clock configuration 2 register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_CLK2_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	switch (GetBitsFromChar(rxData, 4, 0)) // OSR[3:0]; R/W; Reset 6h; Oversampling ratio
	{
		/*
		 * These bits set the OSR to create the ADC output data rate, fDATA.
		 * */
		case 0:
			// fDATA = fMOD / 4096
			// OSR = 4096
			// fDATA at 2.048-MHz fMOD (kHz) = 0.500
			// fDATA at 4.096-MHz fMOD (kHz) = 1.000
			// fDATA at 4-MHz fMOD (kHz) = 0.977
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 4096\n");
			break;
		case 1:
			// fDATA = fMOD / 2048
			// OSR = 2048
			// fDATA at 2.048-MHz fMOD (kHz) = 1.000
			// fDATA at 4.096-MHz fMOD (kHz) = 2.000
			// fDATA at 4-MHz fMOD (kHz) = 1.953
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 2048\n");
			break;
		case 2:
			// fDATA = fMOD / 1024
			// OSR = 1024
			// fDATA at 2.048-MHz fMOD (kHz) = 2.000
			// fDATA at 4.096-MHz fMOD (kHz) = 4.000
			// fDATA at 4-MHz fMOD (kHz) = 3.906
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 1024\n");
			break;
		case 3:
			// fDATA = fMOD / 800
			// OSR = 800
			// fDATA at 2.048-MHz fMOD (kHz) = 2.560
			// fDATA at 4.096-MHz fMOD (kHz) = 5.120
			// fDATA at 4-MHz fMOD (kHz) = 5.000
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 800\n");
			break;
		case 4:
			// fDATA = fMOD / 768
			// OSR = 768
			// fDATA at 2.048-MHz fMOD (kHz) = 2.667
			// fDATA at 4.096-MHz fMOD (kHz) = 5.333
			// fDATA at 4-MHz fMOD (kHz) = 5.208
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 768\n");
			break;
		case 5:
			// fDATA = fMOD / 512
			// OSR = 512
			// fDATA at 2.048-MHz fMOD (kHz) = 4.000
			// fDATA at 4.096-MHz fMOD (kHz) = 8.000
			// fDATA at 4-MHz fMOD (kHz) = 7.813
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 512\n");
			break;
		case 6:
			// fDATA = fMOD / 400
			// OSR = 400
			// fDATA at 2.048-MHz fMOD (kHz) = 5.120
			// fDATA at 4.096-MHz fMOD (kHz) = 10.240
			// fDATA at 4-MHz fMOD (kHz) = 10.000
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 400\n");
			break;
		case 7:
			// fDATA = fMOD / 384
			// OSR = 384
			// fDATA at 2.048-MHz fMOD (kHz) = 5.333
			// fDATA at 4.096-MHz fMOD (kHz) = 10.667
			// fDATA at 4-MHz fMOD (kHz) = 10.417
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 384\n");
			break;
		case 8:
			// fDATA = fMOD / 256
			// OSR = 256
			// fDATA at 2.048-MHz fMOD (kHz) = 8.000
			// fDATA at 4.096-MHz fMOD (kHz) = 16.000
			// fDATA at 4-MHz fMOD (kHz) = 15.625
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 256\n");
			break;
		case 9:
			// fDATA = fMOD / 200
			// OSR = 200
			// fDATA at 2.048-MHz fMOD (kHz) = 10.240
			// fDATA at 4.096-MHz fMOD (kHz) = 20.480
			// fDATA at 4-MHz fMOD (kHz) = 20.000
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 200\n");
			break;
		case 10:
			// fDATA = fMOD / 192
			// OSR = 192
			// fDATA at 2.048-MHz fMOD (kHz) = 10.667
			// fDATA at 4.096-MHz fMOD (kHz) = 21.333
			// fDATA at 4-MHz fMOD (kHz) = 20.833
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 192\n");
			break;
		case 11:
			// fDATA = fMOD / 128
			// OSR = 128
			// fDATA at 2.048-MHz fMOD (kHz) = 16.000
			// fDATA at 4.096-MHz fMOD (kHz) = 32.000
			// fDATA at 4-MHz fMOD (kHz) = 31.250
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 128\n");
			break;
		case 12:
			// fDATA = fMOD / 96
			// OSR = 96
			// fDATA at 2.048-MHz fMOD (kHz) = 21.333
			// fDATA at 4.096-MHz fMOD (kHz) = 42.667
			// fDATA at 4-MHz fMOD (kHz) = 41.667
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 96\n");
			break;
		case 13:
			// fDATA = fMOD / 64
			// OSR = 64
			// fDATA at 2.048-MHz fMOD (kHz) = 32.000
			// fDATA at 4.096-MHz fMOD (kHz) = 64.000
			// fDATA at 4-MHz fMOD (kHz) = 62.500
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 64\n");
			break;
		case 14:
			// fDATA = fMOD / 48
			// OSR = 48
			// fDATA at 2.048-MHz fMOD (kHz) = 42.667
			// fDATA at 4.096-MHz fMOD (kHz) = 85.333
			// fDATA at 4-MHz fMOD (kHz) = 83.333
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 48\n");
			break;
		case 15:
			// fDATA = fMOD / 32
			// OSR = 32
			// fDATA at 2.048-MHz fMOD (kHz) = 64.000
			// fDATA at 4.096-MHz fMOD (kHz) = 128.000
			// fDATA at 4-MHz fMOD (kHz) = 125.000
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "OSR[3:0]; R/W; Reset 6h; Oversampling ratio: OSR = 32\n");
			break;
	}

	// Bit 4 reserved. Always write 0.

	switch (GetBitsFromChar(rxData, 3, 5)) // ICLK_DIV[2:0]; R/W; Reset 4h; ICLK divide ratio
	{
		/*
		 * These bits set the divider ratio to generate the ADC
		 * modulator clock, fMOD, from the fICLK signal.
		 * */
        case 0:
			// Reserved
			break;
		case 1:
			// fMOD = fICLK / 2
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ICLK_DIV[2:0]; R/W; Reset 4h; ICLK divide ratio: fMOD = fICLK / 2\n");
			break;
		case 2:
			// fMOD = fICLK / 4
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ICLK_DIV[2:0]; R/W; Reset 4h; ICLK divide ratio: fMOD = fICLK / 4\n");
			break;
		case 3:
			// fMOD = fICLK / 6
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ICLK_DIV[2:0]; R/W; Reset 4h; ICLK divide ratio: fMOD = fICLK / 6\n");
			break;
		case 4:
			// fMOD = fICLK / 8
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ICLK_DIV[2:0]; R/W; Reset 4h; ICLK divide ratio: fMOD = fICLK / 8\n");
			break;
		case 5:
			// fMOD = fICLK / 10
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ICLK_DIV[2:0]; R/W; Reset 4h; ICLK divide ratio: fMOD = fICLK / 10\n");
			break;
		case 6:
			// fMOD = fICLK / 12
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ICLK_DIV[2:0]; R/W; Reset 4h; ICLK divide ratio: fMOD = fICLK / 12\n");
			break;
		case 7:
			// fMOD = fICLK / 14
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ICLK_DIV[2:0]; R/W; Reset 4h; ICLK divide ratio: fMOD = fICLK / 14\n");
			break;
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul ADC_ENA: ADC Channel Enable Register
 * (address = 0Fh).
 * This register controls the enabling of ADC channels.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului ADC_ENA
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_ADC_ENA(unsigned char* readData)
{
    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ADC_ENA: ADC channel enable register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_ADC_ENA_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	switch (GetBitsFromChar(rxData, 4, 0)) // ENA[3:0]; R/W; Reset 0h; Enable ADC channels
	{
		/*
		 * These bits power-up or power-down the ADC channels.
		 * Note that this setting is global for all channels.
		 * */
        case 0:
			// All ADC channels powered down
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ENA[3:0]; R/W; Reset 0h; Enable ADC channels: All ADC channels powered down\n");
			break;
		case 15:
			// All ADC channels powered up
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ENA[3:0]; R/W; Reset 0h; Enable ADC channels: All ADC channels powered up\n");
			break;
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul ADC1: ADC Channel 1 Digital Gain Configuration Register
 * (address = 11h).
 * This register controls the digital gain setting for the ADC channel 1.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului ADC1
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_ADC1(unsigned char* readData)
{
    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ADC1: ADC channel 1 digital gain configuration register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_ADC1_ADDRESS, &rxData) == false)
    {
        *readData = rxData;
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	switch (GetBitsFromChar(rxData, 3, 0)) // GAIN1_[2:0]; R/W; Reset 0h; Gain control (digital scaling)
	{
		/*
		 * These bits determine the digital gain of the ADC output.
		 * */
        case 0:
			// Gain = 1
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN1_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 1\n");
			break;
		case 1:
			// Gain = 2
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN1_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 2\n");
			break;
		case 2:
			// Gain = 4
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN1_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 4\n");
			break;
		case 3:
			// Gain = 8
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN1_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 8\n");
			break;
		case 4:
			// Gain = 16
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN1_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 16\n");
			break;
		// Values 5, 6, 7 reserved.
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul ADC2: ADC Channel 2 Digital Gain Configuration Register
 * (address = 12h).
 * This register controls the digital gain setting for the ADC channel 2.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului ADC2
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_ADC2(unsigned char* readData)
{
    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ADC2: ADC channel 2 digital gain configuration register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_ADC2_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	switch (GetBitsFromChar(rxData, 3, 0)) // GAIN2_[2:0]; R/W; Reset 0h; Gain control (digital scaling)
	{
		/*
		 * These bits determine the digital gain of the ADC output.
		 * */
        case 0:
			// Gain = 1
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN2_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 1\n");
			break;
		case 1:
			// Gain = 2
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN2_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 2\n");
			break;
		case 2:
			// Gain = 4
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN2_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 4\n");
			break;
		case 3:
			// Gain = 8
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN2_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 8\n");
			break;
		case 4:
			// Gain = 16
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN2_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 16\n");
			break;
		// Values 5, 6, 7 reserved.
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
    return true;
}

/**
 * @brief Citeste registrul ADC3: ADC Channel 3 Digital Gain Configuration Register
 * (address = 13h).
 * This register controls the digital gain setting for the ADC channel 3.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului ADC3
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_ADC3(unsigned char* readData)
{
    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ADC3: ADC channel 3 digital gain configuration register status:\n");

    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_ADC3_ADDRESS, &rxData) == false)
    {
        return false;
    }

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	switch (GetBitsFromChar(rxData, 3, 0)) // GAIN3_[2:0]; R/W; Reset 0h; Gain control (digital scaling)
	{
		/*
		 * These bits determine the digital gain of the ADC output.
		 * */
        case 0:
			// Gain = 1
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN3_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 1\n");
			break;
		case 1:
			// Gain = 2
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN3_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 2\n");
			break;
		case 2:
			// Gain = 4
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN3_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 4\n");
			break;
		case 3:
			// Gain = 8
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN3_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 8\n");
			break;
		case 4:
			// Gain = 16
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN3_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 16\n");
			break;
		// Values 5, 6, 7 reserved.
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
	return true;
}

/**
 * @brief Citeste registrul ADC4: ADC Channel 4 Digital Gain Configuration Register
 * (address = 14h).
 * This register controls the digital gain setting for the ADC channel 4.
 *
 * @param readData Buffer in care se stocheaza valoarea registrului ADC4
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_ADC4(unsigned char* readData)
{
	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "ADC4: ADC channel 4 digital gain configuration register status:\n");

	unsigned char rxData;
	if (ADS131A04_rreg(ADS131A04_ADC4_ADDRESS, &rxData) == false)
	{
	    return false;
	}

#ifdef ADS131A04_INTERPRET_REGISTER_VALUES
	switch (GetBitsFromChar(rxData, 3, 0)) // GAIN4_[2:0]; R/W; Reset 0h; Gain control (digital scaling)
	{
		/*
		 * These bits determine the digital gain of the ADC output.
		 * */
		case 0:
			// Gain = 1
			//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN4_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 1\n");
			break;
		case 1:
			// Gain = 2
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN4_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 2\n");
			break;
		case 2:
			// Gain = 4
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN4_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 4\n");
			break;
		case 3:
			// Gain = 8
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN4_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 8\n");
			break;
		case 4:
			// Gain = 16
		    //if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "GAIN4_[2:0]; R/W; Reset 0h; Gain control (digital scaling): Gain 16\n");
			break;
		// Values 5, 6, 7 reserved.
	}

	//if (ADS131A04_getStatus == true) strcat(ADS131A04_status, "\n");
#endif

	*readData = rxData;
	return true;
}

/**
 * @brief Citeste starea ADC-ului.
 *
 * @return Starea ADC-ului sub forma de string
**/
const char* ADS131A04_getStatus(void)
{
	//ADS131A04_getStatus = true;
	strcpy(ADS131A04_status, "ADC status:\n\n");
	//unsigned char status;
	//ADS131A04_read_STAT_1(&status); // Exemplu utilizare

	/*ADS131A04_read_STAT_1();
	//ADS131A04_read_RevisionID(); // Comentat pentru optimizare (sprintf consuma multe resurse)
	ADS131A04_read_STAT_P();
	ADS131A04_read_STAT_N();
	ADS131A04_read_STAT_S();
	//ADS131A04_read_ErrorCount(); // Comentat pentru optimizare (sprintf consuma multe resurse)
	ADS131A04_read_STAT_M2();
	ADS131A04_read_A_SYS_CFG();
	ADS131A04_read_D_SYS_CFG();
	ADS131A04_read_CLK1();
	ADS131A04_read_CLK2();
	ADS131A04_read_ADC_ENA();
	ADS131A04_read_ADC1();
	ADS131A04_read_ADC2();
	ADS131A04_read_ADC3();
	ADS131A04_read_ADC4();*/

	//ADS131A04_getStatus = false;

	return ADS131A04_status;
}

/**
 * @brief RREG: Read a Single Register. The RREG command reads register data
 * from the ADC. RREG is a 16-bit command containing the command, the register
 * address, and the number of registers to be to read. The command details are
 * shown below:
 * 		First byte: 001a aaaa, where a aaaa is the starting register address.
 * 		Second byte: nnnn nnnn, where nnnn nnnn is the number of registers to
 * 		read minus one (n-1).
 * The ADC executes the command upon completion of the data frame and the
 * register data transmission begins on the first device word of the following
 * data frame. The command status response differs depending on whether a
 * single register or multiple registers are read back. For a single register
 * read, the 16-bit response contains an 8-bit acknowledgment byte with the
 * register address and an 8-bit data byte with the register content.
 *
 * @param registerAddress Adresa registrului care se citeste
 * @param readData Buffer in care se stocheaza valoarea registrului citit
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_rreg(unsigned char registerAddress, unsigned char* readData)
{
    int txData = 0x2000 | (registerAddress << 8);
    if (adcChannelsEnabled == false)
    {
        ADS131A04_writeInt((txData << 16), 4);
    }
    else
    {
        ADS131A04_writeInt((txData << 16), 20);
    }
    unsigned char *rxData;
    if (adcChannelsEnabled == false)
    {
        rxData = ADS131A04_read(4);
    }
    else
    {
        rxData = ADS131A04_read(20);
    }
    if (*rxData == (0x20 | registerAddress))
    {
        // Read register command successful
        *readData = *(rxData + 1);
        return true;
    }
    else
    {
        // Read register command unsuccessful
        return false;
    }
}

/**
 * @brief WREG: Write Single Register. The WREG command writes data to a
 * single register. The single register write command is a two-byte command
 * containing the address and the data to write to the address. The command
 * details are shown below:
 * 		First byte: 010a aaaa, where a aaaa is the register address.
 * 		Second byte: dddd dddd, where dddd dddd is the data to write to the
 * 		address.
 * The resulting command status response is a register read back from the
 * updated register.
 *
 * @param registerAddress Adresa registrului care se scrie
 * @param data Valoarea care se scrie in registru
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_wreg(unsigned char registerAddress, unsigned char data/*, unsigned char* readData*/)
{
	int txData = 0x4000 | (registerAddress << 8) | data;
	if (adcChannelsEnabled == false)
	{
	    ADS131A04_writeInt((txData << 16), 4);
	}
	else
	{
	    ADS131A04_writeInt((txData << 16), 20);
	}

    unsigned char *rxData = (unsigned char*)calloc(20, sizeof(unsigned char));
	if (adcChannelsEnabled == false)
	{
	    rxData = ADS131A04_read(4);
	}
	else
	{
	    rxData = ADS131A04_read(20);
	}
	if (*rxData == (0x20 | registerAddress))
	{
		// Write register command successful
	    //*readData = *(rxData + 1);
	    free(rxData);
		return true;
	}
	else
	{
		// Write register command unsuccessful
	    free(rxData);
		return false;
	}
}

/**
 * @brief Transmite date prin SPI.
 *
 * @param data Valoarea intreaga care se transmite ADC-ului
 * @param bytesNumber Numarul de octeti din care este format intregul
 *
 * @return Numarul de octeti transmisi
**/
unsigned char ADS131A04_writeInt(int data, unsigned char bytesNumber)
{
    MAP_GPIO_setOutputLowOnPin(GPIO_PORT_P1, GPIO_PIN4); // !ADC_SPI_CS = 0

	int byte = 0, i = 3;
	for(byte = bytesNumber - 1; byte >= 0; byte--) // MSB first
	{
	    // Polling to see if the TX buffer is ready
        //while (!(MAP_SPI_getInterruptStatus(EUSCI_B0_BASE,EUSCI_B_SPI_TRANSMIT_INTERRUPT)));

        /*while (adcTxFlag == false) // La frecvente mari nu se mai poate folosi aceasta metoda
        {
            //MAP_PCM_gotoLPM0InterruptSafe(); // La frecvente mari nu se mai poate folosi metoda cu sleep
            //MAP_Interrupt_enableMaster();
        }
        adcTxFlag = false;*/

        // Transmitting data to slave
        if (bytesNumber <= 4)
        {
            //MAP_SPI_transmitData(EUSCI_B0_BASE, GetByte(data, byte));
            EUSCI_B0->TXBUF = GetByte(data, byte); // Este mai rapid
        }
        else
        {
            if (i >= 0)
            {
                //MAP_SPI_transmitData(EUSCI_B0_BASE, GetByte(data, i)); // Transmite intai octeti comenzii, apoi 0
                EUSCI_B0->TXBUF = GetByte(data, i); // Este mai rapid
                i--;
            }
            else
            {
                //MAP_SPI_transmitData(EUSCI_B0_BASE, GetByte(data, 0));
                EUSCI_B0->TXBUF = GetByte(data, 0); // Este mai rapid
            }
        }

        while (!(EUSCI_B0->IFG & UCRXIFG));
        //__delay_cycles(8);
        // La frecvente mari nu se mai poate folosi aceasta metoda
		/*while (adcRxFlag == false) // Goleste buffer-ul de un octetul receptionat ca raspuns la comanda anterioara
        {
            //MAP_PCM_gotoLPM0InterruptSafe(); // La frecvente mari nu se mai poate folosi metoda cu sleep
            //MAP_Interrupt_enableMaster();
        }
		adcRxFlag = false; // Este necesara doar setarea flag-ului
		*/
        //MAP_SPI_receiveData(EUSCI_B0_BASE); // Nu este necesar
	}

	MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P1, GPIO_PIN4); // !ADC_SPI_CS = 1

	return bytesNumber;
}

/**
 * @brief Citeste date prin SPI.
 *
 * @param bytesNumber Numarul de octeti care se citeste
 *
 * @return Pointer catre bufferul de receptie
**/
unsigned char* ADS131A04_read(unsigned char bytesNumber)
{
    MAP_GPIO_setOutputLowOnPin(GPIO_PORT_P1, GPIO_PIN4); // !ADC_SPI_CS = 0

	int byte = 0;
	int count = bytesNumber;
	if (bytesNumber > ADS131A04_MAX_RX_BYTES) count = ADS131A04_MAX_RX_BYTES;

	for(byte = 0; byte < count; byte++)
	{
        /*while (adcTxFlag == false) // La frecvente mari nu se mai poate aceasta metoda
        {
            //MAP_PCM_gotoLPM0InterruptSafe(); // La frecvente mari nu se mai poate folosi metoda cu sleep
            //MAP_Interrupt_enableMaster();
        }
        adcTxFlag = false;*/

        //MAP_SPI_transmitData(EUSCI_B0_BASE, 0x00); // Dummy write to clock out data; genereaza clock-ul necesar pentru citirea datelor
        EUSCI_B0->TXBUF = 0x00; // Este mai rapid

        while (!(EUSCI_B0->IFG & UCRXIFG));
        //__delay_cycles(8);
        /*while (adcRxFlag == false) // La frecvente mari nu se mai poate folosi aceasta metoda
        {
            //MAP_PCM_gotoLPM0InterruptSafe(); // La frecvente mari nu se mai poate folosi metoda cu sleep
            //MAP_Interrupt_enableMaster();
        }
        adcRxFlag = false;*/

        //rxData[byte] = MAP_SPI_receiveData(EUSCI_B0_BASE);
        rxData[byte] = EUSCI_B0->RXBUF; // Este mai rapid
	}

	MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P1, GPIO_PIN4); // !ADC_SPI_CS = 1

	return rxData;
}

/**
 * @brief Seteaza un bit de registru
 *
 * @param registerAddress Adresa registrului care se modifica
 * @param value Valoarea care se scrie
 * @param mask Masca folosita pentru scrierea bitului
 * @param shift Deplasarea folosita pentru scrierea bitului
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_writeRegisterBit(unsigned char registerAddress, unsigned char value, int mask, int shift)
{
    unsigned char rxData;
    if (ADS131A04_rreg(registerAddress, &rxData) == false)
    {
        return false;
    }

    rxData &= ~(mask << shift); // Extrage valorile initiale
    rxData |= (value << shift); // Seteaza bitul

    if (ADS131A04_wreg(registerAddress, rxData) == false)
    {
        return false;
    }

    return true;
}

/**
 * @brief Citeste un bit de registru
 *
 * @param registerAddress Adresa registrului care se citeste
 * @param value Buffer in care se stocheaza valoarea citita
 * @param mask Masca folosita pentru citirea bitului
 * @param shift Deplasarea folosita pentru citirea bitului
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_readRegisterBit(unsigned char registerAddress, bool* value, int mask, int shift)
{
    unsigned char rxData;
    if (ADS131A04_rreg(registerAddress, &rxData) == false)
    {
        return false;
    }

    rxData &= (mask << shift); // Extrage bitul
    *value = (rxData >> shift);  // Scrie bitul in variabila de value

    return true;
}

/**
 * @brief Fault command.
 * This bit indicates that a received command is not recognized as valid and
 * the command is ignored. This bit auto-clears on a STAT_1 data transfer,
 * unless the condition remains.
 * When in a locked state, this bit is set if any command other than LOCK,
 * UNLOCK, NULL, or RREGS is written to the device.
 * 0 : No fault has occurred
 * 1 : Possible invalid command is ignored
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_OPC(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_1_ADDRESS, value, ADS131A04_STAT_1_F_OPC_MASK, ADS131A04_STAT_1_F_OPC_SHIFT))
            return false;
    return true;
}

/**
 * @brief Fault SPI.
 * This bit indicates that one of the status bits in the STAT_S register is set.
 * Read the STAT_S register to clear the bit.
 * 0 : No fault has occurred
 * 1 : A bit in the STAT_S register is set high
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_SPI(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_1_ADDRESS, value, ADS131A04_STAT_1_F_SPI_MASK, ADS131A04_STAT_1_F_SPI_SHIFT))
            return false;
    return true;
}


/**
 * @brief Fault ADC input.
 * This bit indicates that one of the ADC input fault detection bits in the
 * STAT_P or STAT_N register is set. Read the STAT_P and STAT_N
 * registers to clear the bit.
 * 0 : No fault has occurred
 * 1 : A bit in the STAT_P or STAT_N register is set high
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_ADCIN(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_1_ADDRESS, value, ADS131A04_STAT_1_F_ADCIN_MASK, ADS131A04_STAT_1_F_ADCIN_SHIFT))
            return false;
    return true;
}


/**
 * @brief Watchdog timer timeout.
 * This bit indicates if the watchdog timer times out before a new data frame
 * transfer occurs.
 * 0 : No fault has occurred
 * 1 : Timer has run out (resets following register read back)
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_WDT(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_1_ADDRESS, value, ADS131A04_STAT_1_F_WDT_MASK, ADS131A04_STAT_1_F_WDT_SHIFT))
            return false;
    return true;
}


/**
 * @brief Fault resynchronization.
 * This bit is set whenever the signal path is momentarily reset resulting from
 * a DRDY synchronization event.
 * 0 : Devices are in sync
 * 1 : Signal path is momentarily reset to maintain synchronization
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_RESYNC(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_1_ADDRESS, value, ADS131A04_STAT_1_F_RESYNC_MASK, ADS131A04_STAT_1_F_RESYNC_SHIFT))
            return false;
    return true;
}


/**
 * @brief Fault data ready.
 * This bit is set if data shifted out from the previous result are not complete
 * by the time new ADC data are ready. The ADC DRDY line pulses,
 * indicating that new data are available and overwrite the current data. This
 * bit auto-clears on a STAT_1 transfer, unless the condition remains.
 * 0 : Data read back complete before new data update
 * 1 : New data update during DOUT data transmission
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_DRDY(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_1_ADDRESS, value, ADS131A04_STAT_1_F_DRDY_MASK, ADS131A04_STAT_1_F_DRDY_SHIFT))
            return false;
    return true;
}


/**
 * @brief Fault DIN check.
 * This bit is set if either of the following conditions are detected:
 *      Uncorrectable hamming error correction state is determined for any
 *      DIN word transfer when hamming code is enabled.
 *      CRC check word on DIN fails. The input command that triggered this
 *      error is ignored.
 * This bit auto-clears on a STAT_S transfer, unless the condition remains.
 * 0 : No error in DIN transmission
 * 1 : DIN transmission error
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_CHECK(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_1_ADDRESS, value, ADS131A04_STAT_1_F_CHECK_MASK, ADS131A04_STAT_1_F_CHECK_SHIFT))
            return false;
    return true;
}


/**
 * @brief AIN4P threshold detect.
 * 0 : The channel 4 positive input pin does not exceed the set threshold
 * 1 : The channel 4 positive input pin exceeds the set threshold
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_IN4P(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_P_ADDRESS, value, ADS131A04_STAT_P_F_IN4P_MASK, ADS131A04_STAT_P_F_IN4P_SHIFT))
            return false;
    return true;
}


/**
 * @brief AIN3P threshold detect.
 * 0 : The channel 3 positive input pin does not exceed the set threshold
 * 1 : The channel 3 positive input pin exceeds the set threshold
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_IN3P(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_P_ADDRESS, value, ADS131A04_STAT_P_F_IN3P_MASK, ADS131A04_STAT_P_F_IN3P_SHIFT))
            return false;
    return true;
}


/**
 * @brief AIN2P threshold detect.
 * 0 : The channel 2 positive input pin does not exceed the set threshold
 * 1 : The channel 2 positive input pin exceeds the set threshold
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_IN2P(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_P_ADDRESS, value, ADS131A04_STAT_P_F_IN2P_MASK, ADS131A04_STAT_P_F_IN2P_SHIFT))
            return false;
    return true;
}


/**
 * @brief AIN1P threshold detect.
 * 0 : The channel 1 positive input pin does not exceed the set threshold
 * 1 : The channel 1 positive input pin exceeds the set threshold
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_IN1P(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_P_ADDRESS, value, ADS131A04_STAT_P_F_IN1P_MASK, ADS131A04_STAT_P_F_IN1P_SHIFT))
            return false;
    return true;
}


/**
 * @brief AIN4N threshold detect.
 * 0 : The channel 4 negative input pin does not exceed the set threshold
 * 1 : The channel 4 negative input pin exceeds the set threshold
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_IN4N(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_N_ADDRESS, value, ADS131A04_STAT_N_F_IN4N_MASK, ADS131A04_STAT_N_F_IN4N_SHIFT))
            return false;
    return true;
}


/**
 * @brief AIN3N threshold detect
 * 0 : The channel 3 negative input pin does not exceed the set threshold
 * 1 : The channel 3 negative input pin exceeds the set threshold
 *
 * @param value Buffer in care se stocheaza valoarea citita.
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat.
**/
bool ADS131A04_read_F_IN3N(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_N_ADDRESS, value, ADS131A04_STAT_N_F_IN3N_MASK, ADS131A04_STAT_N_F_IN3N_SHIFT))
            return false;
    return true;
}


/**
 * @brief AIN2N threshold detect.
 * 0 : The channel 2 negative input pin does not exceed the set threshold
 * 1 : The channel 2 negative input pin exceeds the set threshold
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_IN2N(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_N_ADDRESS, value, ADS131A04_STAT_N_F_IN2N_MASK, ADS131A04_STAT_N_F_IN2N_SHIFT))
            return false;
    return true;
}


/**
 * @brief AIN1N threshold detect.
 * 0 : The channel 1 negative input pin does not exceed the set threshold
 * 1 : The channel 1 negative input pin exceeds the set threshold
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_IN1N(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_N_ADDRESS, value, ADS131A04_STAT_N_F_IN1N_MASK, ADS131A04_STAT_N_F_IN1N_SHIFT))
            return false;
    return true;
}


/**
 * @brief ADC startup fault.
 * This bit indicates if an error is detected during power-up. This bit clears
 * only when power is recycled.
 * 0 : No fault is occurred
 * 1 : A fault has occurred
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_STARTUP(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_S_ADDRESS, value, ADS131A04_STAT_S_F_STARTUP_MASK, ADS131A04_STAT_S_F_STARTUP_SHIFT))
            return false;
    return true;
}


/**
 * @brief Chip-select fault.
 * This bit is set if CS transitions when the SCLK pin is high. This bit autoclears
 * on a STAT_S transfer, unless the condition remains.
 * 0 : CS is asserted or deasserted when SCLK is low
 * 1 : CS is asserted or deasserted when SCLK is high
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_CS(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_S_ADDRESS, value, ADS131A04_STAT_S_F_CS_MASK, ADS131A04_STAT_S_F_CS_SHIFT))
            return false;
    return true;
}


/**
 * @brief Frame fault.
 * This bit is set if the device detects that not enough SCLK cycles are sent in
 * a data frame for the existing mode of operation. This bit auto-clears on a
 * STAT_S transfer, unless the condition remains.
 * 0 : Enough SCLKs are sent per frame
 * 1 : Not enough SCLKs are sent per frame
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_F_FRAME(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_STAT_S_ADDRESS, value, ADS131A04_STAT_S_F_FRAME_MASK, ADS131A04_STAT_S_F_FRAME_SHIFT))
            return false;
    return true;
}


/**
 * @brief Negative charge pump enable.
 * This bit enables the negative charge pump when using a 3.0-V to 3.45-V
 * unipolar power supply.
 * 0 : Negative charge pump powered down
 * 1 : Negative charge pump enabled
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_VNCPEN(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_VNCPEN_MASK, ADS131A04_A_SYS_CFG_VNCPEN_SHIFT))
            return false;
    return true;
}


/**
 * @brief Negative charge pump enable.
 * This bit enables the negative charge pump when using a 3.0-V to 3.45-V
 * unipolar power supply.
 * 0 : Negative charge pump powered down
 * 1 : Negative charge pump enabled
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_VNCPEN(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_VNCPEN_MASK, ADS131A04_A_SYS_CFG_VNCPEN_SHIFT))
            return false;
    return true;
}


/**
 * @brief High-resolution mode.
 * This bit selects between high-resolution and low-power mode.
 * 0 : Low-power mode
 * 1 : High-resolution mode
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_HRM(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_HRM_MASK, ADS131A04_A_SYS_CFG_HRM_SHIFT))
            return false;
    return true;
}


/**
 * @brief High-resolution mode.
 * This bit selects between high-resolution and low-power mode.
 * 0 : Low-power mode
 * 1 : High-resolution mode
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_HRM(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_HRM_MASK, ADS131A04_A_SYS_CFG_HRM_SHIFT))
            return false;
    return true;
}


/**
 * @brief REFP reference voltage level.
 * This bit determines the REFP reference voltage level when using the
 * internal reference.
 * 0 : REFP is set to 2.442 V
 * 1 : REFP is set to 4.0 V
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_VREF_4V(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_VREF_4V_MASK, ADS131A04_A_SYS_CFG_VREF_4V_SHIFT))
            return false;
    return true;
}


/**
 * @brief REFP reference voltage level.
 * This bit determines the REFP reference voltage level when using the
 * internal reference.
 * 0 : REFP is set to 2.442 V
 * 1 : REFP is set to 4.0 V
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_VREF_4V(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_VREF_4V_MASK, ADS131A04_A_SYS_CFG_VREF_4V_SHIFT))
            return false;
    return true;
}


/**
 * @brief Enable internal reference.
 * This bit connects the internal reference voltage to the reference buffer to
 * use the internal reference
 * 0 : External reference voltage
 * 1 : Internal reference voltage enabled
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_INT_REFEN(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_INT_REFEN_MASK, ADS131A04_A_SYS_CFG_INT_REFEN_SHIFT))
            return false;
    return true;
}


/**
 * @brief Enable internal reference.
 * This bit connects the internal reference voltage to the reference buffer to
 * use the internal reference
 * 0 : External reference voltage
 * 1 : Internal reference voltage enabled
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_INT_REFEN(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_INT_REFEN_MASK, ADS131A04_A_SYS_CFG_INT_REFEN_SHIFT))
            return false;
    return true;
}


/**
 * @brief Fault detect comparator threshold.
 * These bits determine the fault detect comparator threshold level settings.
 * When VNCPEN = 0, AVDD and AVSS are used for the high and low
 * threshold.
 * When VNCPEN = 1, AVDD is used for the high threshold value. A 1.5-V
 * supply, generated from the negative charge pump, is used for the low
 * threshold value.
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_COMP_TH(ADS131A04_COMP_TH_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_A_SYS_CFG_ADDRESS, value, ADS131A04_A_SYS_CFG_COMP_TH_MASK, ADS131A04_A_SYS_CFG_COMP_TH_SHIFT))
            return false;
    return true;
}


/**
 * @brief Watchdog timer enable.
 * This bit enables the watchdog timeout counter when set.
 * Issue a hardware or software reset when disabling the watchdog timer for
 * internal device synchronization.
 * 0 : Watchdog disabled
 * 1 : Watchdog enabled
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_WDT_EN(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_WDT_EN_MASK, ADS131A04_D_SYS_CFG_WDT_EN_SHIFT))
            return false;
    return true;
}


/**
 * @brief Watchdog timer enable.
 * This bit enables the watchdog timeout counter when set.
 * Issue a hardware or software reset when disabling the watchdog timer for
 * internal device synchronization.
 * 0 : Watchdog disabled
 * 1 : Watchdog enabled
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_WDT_EN(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_WDT_EN_MASK, ADS131A04_D_SYS_CFG_WDT_EN_SHIFT))
            return false;
    return true;
}


/**
 * @brief CRC mode select.
 * This bit determines which bits in the frame the CRC is valid for.
 * 0 : CRC is valid on only the device words being sent and received
 * 1 : CRC is valid on all bits received and transmitted
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_CRC_MODE(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_CRC_MODE_MASK, ADS131A04_D_SYS_CFG_CRC_MODE_SHIFT))
            return false;
    return true;
}


/**
 * @brief CRC mode select.
 * This bit determines which bits in the frame the CRC is valid for.
 * 0 : CRC is valid on only the device words being sent and received
 * 1 : CRC is valid on all bits received and transmitted
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_CRC_MODE(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_CRC_MODE_MASK, ADS131A04_D_SYS_CFG_CRC_MODE_SHIFT))
            return false;
    return true;
}


/**
 * @brief DONE delay.
 * These bits configure the time before the device asserts DONE after the
 * LSB is shifted out.
 * 00 : equal or greater than 6-ns delay
 * 01 : equal or greater than 8-ns delay
 * 10 : equal or greater than 10-ns delay
 * 11 : equal or greater than 12-ns delay
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_DNDLY(ADS131A04_DLY_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_DNDLY_MASK, ADS131A04_D_SYS_CFG_DNDLY_SHIFT))
            return false;
    return true;
}


/**
 * @brief Hi-Z delay.
 * These bits configure the time that the device asserts Hi-Z on DOUT after
 * the LSB of the data frame is shifted out.
 * 00 : equal or greater than 6-ns delay
 * 01 : equal or greater than 8-ns delay
 * 10 : equal or greater than 10-ns delay
 * 11 : equal or greater than 12-ns delay
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_HIZDLY(ADS131A04_DLY_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_HIZDLY_MASK, ADS131A04_D_SYS_CFG_HIZDLY_SHIFT))
            return false;
    return true;
}


/**
 * @brief Fixed word size enable.
 * This bit sets the data frame size.
 * 0 : Device words per data frame depends on whether the CRC and ADCs
 * are enabled
 * 1 : Fixed six device words per frame for the ADS131A04 or fixed four
 * device words per data frame for the ADS131A02
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_FIXED(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_FIXED_MASK, ADS131A04_D_SYS_CFG_FIXED_SHIFT))
            return false;
    return true;
}


/**
 * @brief Fixed word size enable.
 * This bit sets the data frame size.
 * 0 : Device words per data frame depends on whether the CRC and ADCs
 * are enabled
 * 1 : Fixed six device words per frame for the ADS131A04 or fixed four
 * device words per data frame for the ADS131A02
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_FIXED(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_FIXED_MASK, ADS131A04_D_SYS_CFG_FIXED_SHIFT))
            return false;
    return true;
}


/**
 * @brief Cyclic redundancy check enable.
 * This bit enables the CRC data word for both the DIN and DOUT data frame
 * transfers. When enabled, DIN commands must pass the CRC checks to be
 * recognized by the device.
 * 0 : CRC disabled
 * 1 : CRC enabled
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_CRC_EN(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_CRC_EN_MASK, ADS131A04_D_SYS_CFG_CRC_EN_SHIFT))
            return false;
    return true;
}


/**
 * @brief Cyclic redundancy check enable.
 * This bit enables the CRC data word for both the DIN and DOUT data frame
 * transfers. When enabled, DIN commands must pass the CRC checks to be
 * recognized by the device.
 * 0 : CRC disabled
 * 1 : CRC enabled
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_CRC_EN(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_D_SYS_CFG_ADDRESS, value, ADS131A04_D_SYS_CFG_CRC_EN_MASK, ADS131A04_D_SYS_CFG_CRC_EN_SHIFT))
            return false;
    return true;
}


/**
 * @brief ADC clock source.
 * This bit selects the source for ICLK; see the Clock section for more
 * information on ADC clocking.
 * 0 : XTAL1/CLKIN pin or XTAL1/CLKIN and XTAL2 pins
 * 1 : SCLK pin
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_CLKSRC(bool value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_CLK1_ADDRESS, value, ADS131A04_CLK1_CLKSRC_MASK, ADS131A04_CLK1_CLKSRC_SHIFT))
            return false;
    return true;
}


/**
 * @brief ADC clock source.
 * This bit selects the source for ICLK; see the Clock section for more
 * information on ADC clocking.
 * 0 : XTAL1/CLKIN pin or XTAL1/CLKIN and XTAL2 pins
 * 1 : SCLK pin
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_CLKSRC(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_CLK1_ADDRESS, value, ADS131A04_CLK1_CLKSRC_MASK, ADS131A04_CLK1_CLKSRC_SHIFT))
            return false;
    return true;
}


/**
 * @brief CLKIN divider ratio.
 * These bits set the CLKIN divider ratio to generate the internal fICLK
 * frequency. ICLK is used as the fSCLK output when the ADC is operating in
 * an SPI master mode.
 * 000 : Reserved
 * 001 : fICLK = fCLKIN / 2
 * 010 : fICLK = fCLKIN / 4
 * 011 : fICLK = fCLKIN / 6
 * 100 : fICLK = fCLKIN / 8
 * 101 : fICLK = fCLKIN / 10
 * 110 : fICLK = fCLKIN / 12
 * 111 : fICLK = fCLKIN / 14
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_CLK_DIV(ADS131A04_CLK_DIV_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_CLK1_ADDRESS, value, ADS131A04_CLK1_CLK_DIV_MASK, ADS131A04_CLK1_CLK_DIV_SHIFT))
            return false;
    return true;
}


/**
 * @brief ICLK divide ratio.
 * These bits set the divider ratio to generate the ADC modulator clock, fMOD,
 * from the fICLK signal.
 * 000 : Reserved
 * 001 : fMOD = fICLK / 2
 * 010 : fMOD = fICLK / 4
 * 011 : fMOD = fICLK / 6
 * 100 : fMOD = fICLK / 8
 * 101 : fMOD = fICLK / 10
 * 110 : fMOD = fICLK / 12
 * 111 : fMOD = fICLK / 14
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_ICLK_DIV(ADS131A04_ICLK_DIV_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_CLK2_ADDRESS, value, ADS131A04_CLK2_ICLK_DIV_MASK, ADS131A04_CLK2_ICLK_DIV_SHIFT))
            return false;
    return true;
}


/**
 * @brief Oversampling ratio.
 * These bits set the OSR to create the ADC output data rate, fDATA.
 * 0000 : fDATA = fMOD / 4096
 * 0001 : fDATA = fMOD / 2048
 * 0010 : fDATA = fMOD / 1024
 * 0011 : fDATA = fMOD / 800
 * 0100 : fDATA = fMOD / 768
 * 0101 : fDATA = fMOD / 512
 * 0110 : fDATA = fMOD / 400
 * 0111 : fDATA = fMOD / 384
 * 1000 : fDATA = fMOD / 256
 * 1001 : fDATA = fMOD / 200
 * 1010 : fDATA = fMOD / 192
 * 1011 : fDATA = fMOD / 128
 * 1100 : fDATA = fMOD / 96
 * 1101 : fDATA = fMOD / 64
 * 1110 : fDATA = fMOD / 48
 * 1111 : fDATA = fMOD / 32
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_OSR(ADS131A04_OSR_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_CLK2_ADDRESS, value, ADS131A04_CLK2_OSR_MASK, ADS131A04_CLK2_OSR_SHIFT))
            return false;
    return true;
}


/**
 * @brief Enable ADC channels.
 * These bits power-up or power-down the ADC channels. Note that this
 * setting is global for all channels.
 * 0000 : All ADC channels powered down
 * 1111 : All ADC channels powered up
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_ENA(ADS131A04_ENA_Values value)
{
    unsigned char rxData;
    if (ADS131A04_rreg(ADS131A04_ADC_ENA_ADDRESS, &rxData) == false)
    {
        return false;
    }

    rxData &= ~(ADS131A04_ADC_ENA_ENA_MASK << ADS131A04_ADC_ENA_ENA_SHIFT); // Extrage valorile initiale
    rxData |= (value << ADS131A04_ADC_ENA_ENA_SHIFT); // Seteaza bitul

    int txData = 0x4000 | (ADS131A04_ADC_ENA_ADDRESS << 8) | value;
    if (adcChannelsEnabled == false)
    {
        ADS131A04_writeInt((txData << 16), 4);
    }
    else
    {
        ADS131A04_writeInt((txData << 16), 20);
    }

    if (value == 0) // Dupa activarea canalelor, cu CRC dezactivat, trebuie cititi 20 octeti
    {
        adcChannelsEnabled = false;
    }
    else
    {
        adcChannelsEnabled = true;
    }

    unsigned char *rx = (unsigned char*)calloc(20, sizeof(unsigned char));
    if (adcChannelsEnabled == false)
    {
        rx = ADS131A04_read(4);
    }
    else
    {
        rx = ADS131A04_read(20);
    }
    if (*rx == (0x20 | ADS131A04_ADC_ENA_ADDRESS))
    {
        // Write register command successful
        //*readData = *(rx + 1);
        free(rx);
        return true;
    }
    else
    {
        // Write register command unsuccessful
        free(rx);
        return false;
    }
}


/**
 * @brief Gain control (digital scaling).
 * These bits determine the digital gain of the ADC output.
 * 000 : Gain = 1
 * 001 : Gain = 2
 * 010 : Gain = 4
 * 011 : Gain = 8
 * 100 : Gain = 16
 * 101, 110, 111 : Reserved
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_GAIN1(ADS131A04_GAIN_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_ADC1_ADDRESS, value, ADS131A04_ADC1_GAIN1_MASK, ADS131A04_ADC1_GAIN1_SHIFT))
            return false;
    return true;
}


/**
 * @brief Gain control (digital scaling).
 * These bits determine the digital gain of the ADC output.
 * 000 : Gain = 1
 * 001 : Gain = 2
 * 010 : Gain = 4
 * 011 : Gain = 8
 * 100 : Gain = 16
 * 101, 110, 111 : Reserved
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_GAIN1(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_ADC1_ADDRESS, value, ADS131A04_ADC1_GAIN1_MASK, ADS131A04_ADC1_GAIN1_SHIFT))
            return false;
    return true;
}


/**
 * @brief Gain control (digital scaling).
 * These bits determine the digital gain of the ADC output.
 * 000 : Gain = 1
 * 001 : Gain = 2
 * 010 : Gain = 4
 * 011 : Gain = 8
 * 100 : Gain = 16
 * 101, 110, 111 : Reserved
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_GAIN2(ADS131A04_GAIN_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_ADC2_ADDRESS, value, ADS131A04_ADC2_GAIN2_MASK, ADS131A04_ADC2_GAIN2_SHIFT))
            return false;
    return true;
}


/**
 * @brief Gain control (digital scaling).
 * These bits determine the digital gain of the ADC output.
 * 000 : Gain = 1
 * 001 : Gain = 2
 * 010 : Gain = 4
 * 011 : Gain = 8
 * 100 : Gain = 16
 * 101, 110, 111 : Reserved
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_GAIN2(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_ADC2_ADDRESS, value, ADS131A04_ADC2_GAIN2_MASK, ADS131A04_ADC2_GAIN2_SHIFT))
            return false;
    return true;
}


/**
 * @brief Gain control (digital scaling).
 * These bits determine the digital gain of the ADC output.
 * 000 : Gain = 1
 * 001 : Gain = 2
 * 010 : Gain = 4
 * 011 : Gain = 8
 * 100 : Gain = 16
 * 101, 110, 111 : Reserved
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_GAIN3(ADS131A04_GAIN_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_ADC3_ADDRESS, value, ADS131A04_ADC3_GAIN3_MASK, ADS131A04_ADC3_GAIN3_SHIFT))
            return false;
    return true;
}


/**
 * @brief Gain control (digital scaling).
 * These bits determine the digital gain of the ADC output.
 * 000 : Gain = 1
 * 001 : Gain = 2
 * 010 : Gain = 4
 * 011 : Gain = 8
 * 100 : Gain = 16
 * 101, 110, 111 : Reserved
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_GAIN3(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_ADC3_ADDRESS, value, ADS131A04_ADC3_GAIN3_MASK, ADS131A04_ADC3_GAIN3_SHIFT))
            return false;
    return true;
}


/**
 * @brief Gain control (digital scaling).
 * These bits determine the digital gain of the ADC output.
 * 000 : Gain = 1
 * 001 : Gain = 2
 * 010 : Gain = 4
 * 011 : Gain = 8
 * 100 : Gain = 16
 * 101, 110, 111 : Reserved
 *
 * @param value Valoarea setata
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_write_GAIN4(ADS131A04_GAIN_Values value)
{
    if (!ADS131A04_writeRegisterBit(ADS131A04_ADC4_ADDRESS, value, ADS131A04_ADC4_GAIN4_MASK, ADS131A04_ADC4_GAIN4_SHIFT))
            return false;
    return true;
}


/**
 * @brief Gain control (digital scaling).
 * These bits determine the digital gain of the ADC output.
 * 000 : Gain = 1
 * 001 : Gain = 2
 * 010 : Gain = 4
 * 011 : Gain = 8
 * 100 : Gain = 16
 * 101, 110, 111 : Reserved
 *
 * @param value Buffer in care se stocheaza valoarea citita
 *
 * @return true/false daca comanda s-a realizat cu succes/a esuat
**/
bool ADS131A04_read_GAIN4(bool* value)
{
    if (!ADS131A04_readRegisterBit(ADS131A04_ADC4_ADDRESS, value, ADS131A04_ADC4_GAIN4_MASK, ADS131A04_ADC4_GAIN4_SHIFT))
            return false;
    return true;
}

